-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/dbi/adbc/gnu-db-adbc-statement.ads,v $
--  Description     : Ada Database Objects Toplevel package                  --
--  Author          : Michael Erdmann                                        --
--  Created         : 18.1.2002                                              --
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2002/02/23 17:31:10 $
--  Version         : <d>$Revision: 1.3 $</d>
--  Status          : $State: Exp $
--                                                                           --
--  Copyright (C) 2001 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  This software is implemented to work with GNAT, the GNU Ada compiler.    --
--  <p>                                                                      --
--  Functional Description                                                   --
--  ======================                                                   --
--  This is the statement object. The statement object is used to hold the   --
--  information about a SQL statement. A statement object is not connected   --
--  to any data base connection. It is a container for the syntactical       --
--  elements of an SQL statement.                                            --
--                                                                           --
--  A statement like:                                                        --
--                                                                           --
--       select * from employees where name = :myname and empno = :id        --
--                                                                           --
--  is stored in the object in such a way, that the driver is able to        --
--  build a correct query by substituting the named hast variables with the  --
--  correct representations e.g. by adding an '?'.                           --
--  The command is broken up into chunks and arguments as shown below:       --
--                                                                           --
--      chunk 1 - select * from employees where name =                       --
--      arg.  1 - myname                                                     --
--      chunk 2 - and empno =                                                --
--      arg.  2 - id                                                         --
--                                                                           --
--  In order to access these elements, the class provides the operations     --
--  Get_Chunk and Argument.                                                  --
--  </p>                                                                     --
--  Restrictions                                                             --
--  ============                                                             --
--  Only Linux                                                               --
--                                                                           --
--  Contact                                                                  --
--  =======                                                                  --
--  Error reports shall be handled via http://gnade.sourceforge.net          --
--  Features and ideas via: gnade-develop@lists.sourceforge.net              --
--                                                                           --
--  Author contact:                                                          --
--               purl:/net/michael.erdmann                                   --
--                                                                           --
-------------------------------------------------------------------------------
with GNU.DB.ADBC.Connection;
with GNU.DB.ADBC.Hostvariable;
with GNU.DB.ADBC.Driver;
with GNU.DB.ADBC.Row;
use  GNU.DB.ADBC;

package GNU.DB.ADBC.Statement is

   type Object( Con : Connection_ID ) is tagged private;
   type Handle is access all Object'Class;

   Null_Handle : constant Handle       := null;
   Null_ID     : constant Statement_ID := Statement_ID'First;


   ---------------------------------------------------------------------------
   --# <?>Description    :
   --#    Create a stement from the given statement.
   --# Preconditions  :
   --#    None
   --# Postconditions :
   --#
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note</?>           :
   ---------------------------------------------------------------------------
   procedure Prepare(
      Stmt      : in Statement_ID;
      Statement : in String );

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Bind a host variable to a named used in the statement. The actual
   --#    value subsitution is done at execution time of the execute
   --#    method.
   --# Preconditions  :
   --#    R.1 the statement has been prepared
   --#    R.2 the name where the host variable is bound to is contained
   --#        in the command.
   --# Postconditions :
   --#    P.1 - A statement id is allocated to the statement.
   --# Exceptions     :
   --#    Unused_Host_Variable    - R.2 violated
   --#    Bind_table_Overflow     - P.1 is not possible
   --# Note           :
   ---------------------------------------------------------------------------
   procedure Bind(
      Stmt : in Statement_ID;
      Name : in String;
      Var  : Hostvariable.Object'Class );

   Bind_Table_Overflow     : exception;
   Unused_Host_Variable    : exception;
   Host_Variable_Not_Found : exception;

   ---------------------------------------------------------------------------
   --# Description    :
   --# Preconditions  :
   --# Postconditions :
   --# Exceptions     :
   --# Note           :
   ---------------------------------------------------------------------------
   procedure Deallocate(
      Id     : in Statement_ID );

   ---------------------------------------------------------------------------
   --# Description    :
   --# Preconditions  :
   --# Postconditions :
   --# Exceptions     :
   --# Note           :
   --#    *** ONLY FOR DRIVER INTERFACE ***
   ---------------------------------------------------------------------------
   procedure Bind_Host_Variables(
      Stmt : in Statement_ID );

   procedure Retrive_Host_Variables(
      Stmt   : in Statement_ID;
      result : in Row.handle );

   function Query(
      Stmt      : in Statement_ID ) return String;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    The given object handle is stored in a slot of the object table. The
   --#    actual allocation of memory has to be done by the driver.
   --#    This function is intended for usage  by the driver implementations.
   --# Preconditions  :
   --#    R.1  - Handle is not a null handle
   --# Postconditions :
   --#    P.1  - Function returns the so called statement id.
   --# Exceptions     :
   --#    Statement_Table_Overflow - P.1 not possible
   --#    Usage_Error              - R.1
   --# Note           :
   --#    *** ONLY FOR DRIVER INTERFACE ***
   ---------------------------------------------------------------------------
   function Allocate(
      Stmt   : in Handle ) return Statement_ID ;

   Statement_Table_Overflow : exception;
   Usage_Error              : exception;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Add or delete a result set to/from the statement
   --# Preconditions  :
   --# Postconditions :
   --# Exceptions     :
   --# Note           :
   ---------------------------------------------------------------------------
   procedure Delete_Resultset(
      Stmt   : in Statement_ID;
      Result : in Resultset_ID );

   procedure Add_Resultset(
      Stmt   : in Statement_ID;
      Result : in Resultset_ID );

   ---------------------------------------------------------------------------
   --# Description    :
   --# Preconditions  :
   --# Postconditions :
   --# Exceptions     :
   --# Note           :
   ---------------------------------------------------------------------------
   function Get_Handle(
      Stmt   : in Statement_ID ) return Handle;

   function Driver_Handle(
      Stmt   : in Statement_ID ) return Driver.Handle;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Destroy the statement object
   --# Preconditions  :
   --#    R.1 - the object has been created
   --# Postconditions :
   --#    P.1 - The statement object is destroyed all references to it are
   --#          not valid any more.
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------


   ---=====================================================================---
private
   type Object_Data;
   type Object_Data_Access is access Object_Data;

   type Object( Con : Connection_ID ) is tagged record
         Data : Object_Data_Access := null;
      end record;

end GNU.DB.ADBC.Statement;
