/***************************************************************************
 *   Copyright (C) 2007-2012 by Marcel Hasler                              *
 *   mahasler@gmail.com                                                    *
 *                                                                         *
 *   This file is part of KGmailNotifier.                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation, either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the          *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program. If not, see <http://www.gnu.org/licenses/>.  *
 ***************************************************************************/

#include "trayicon.h"

#include <QIcon>
#include <QPainter>
#include <QToolTip>

#include <KColorScheme>
#include <KGlobalSettings>
#include <KIcon>
#include <KLocale>

TrayIcon::TrayIcon(QObject *parent)
    : QObject(parent),
      m_notifierItem(this)
{
    m_notifierItem.setIconByPixmap(KIcon("kgmailnotifier-nomail").pixmap(22));
    m_notifierItem.setCategory(KStatusNotifierItem::Communications);
    m_notifierItem.setStatus(KStatusNotifierItem::Passive);
    m_notifierItem.setToolTip("kgmailnotifier", "KGmailNotifier", "");

    connect(&m_notifierItem, SIGNAL(activateRequested(bool, const QPoint&)),
            this, SLOT(slotActivateRequested(bool, const QPoint&)));
}


TrayIcon::~TrayIcon()
{
}


void TrayIcon::setActive(bool active)
{
    m_notifierItem.setStatus(active ? KStatusNotifierItem::Active : KStatusNotifierItem::Passive);
}


void TrayIcon::setIcon(IconType type, unsigned int number, bool showNumber)
{
    if (type == Alert)
        m_notifierItem.setIconByPixmap(KIcon("kgmailnotifier-alert").pixmap(22));
    else if (type == NoMail)
        m_notifierItem.setIconByPixmap(KIcon("kgmailnotifier-nomail").pixmap(22));
    else
    {
        if (showNumber && number > 0) // Display number on systray
        {
            // The following code is inspired by Akregator's tray icon code
            // and used in order to provide a more consistent systray look

            QPixmap icon = KIcon("kgmailnotifier-newmail").pixmap(22);
            int iconWidth = icon.width();

            QString numberStr = QString::number(number);
            QFont f = KGlobalSettings::generalFont();
            f.setBold(true);

            // Scale the number to fit nicely inside the icon
            float pointSize = f.pointSizeF();
            QFontMetrics fm(f);
            int width = fm.width(numberStr);
            if (width > (iconWidth - 2))
            {
                pointSize *= float(iconWidth - 2) / float(width);
                f.setPointSizeF(pointSize);
            }
            
            // Compose a new icon
            QPixmap finalIcon = icon;
            QPainter p(&finalIcon);
            p.setFont(f);
            KColorScheme scheme(QPalette::Active, KColorScheme::View);

            // Create a lighter background frame for the number
            fm = QFontMetrics(f);
            QRect boundingRect = fm.tightBoundingRect(numberStr);
            boundingRect.setWidth(boundingRect.width() + 2);
            boundingRect.setHeight(boundingRect.height() + 2);
            boundingRect.moveTo((iconWidth - boundingRect.width()) / 2,
                                (iconWidth - boundingRect.height()) / 2);

            // Draw the frame
            p.setOpacity(0.7f);
            p.setBrush(scheme.background(KColorScheme::LinkBackground));
            p.setPen(Qt::NoPen);
            p.drawRoundedRect(boundingRect, 2.0, 2.0);

            // Draw the number
            p.setOpacity(1.0f);
            p.setBrush(Qt::NoBrush);
            p.setPen(Qt::darkBlue);
            p.drawText(finalIcon.rect(), Qt::AlignCenter, numberStr);

            // Display the final icon
            m_notifierItem.setIconByPixmap(QIcon(finalIcon));
        }
        else
            m_notifierItem.setIconByPixmap(KIcon("kgmailnotifier-newmail").pixmap(22));
    }
}


KMenu* TrayIcon::contextMenu() const
{
    return m_notifierItem.contextMenu();
}


void TrayIcon::setToolTip(const QString &text)
{
    m_notifierItem.setToolTipSubTitle(text);
}


void TrayIcon::slotActivateRequested(bool, const QPoint&)
{
    emit clicked();
}
