/***************************************************************************
 *   Copyright (C) 2005-2013 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QEventLoop>

#include <KLocale>
#include <KDebug>

#include "kfileitemext.h"

#include "komparatoremptydirectoriesjob.h"

#include <version.h>

KomparatorEmptyDirectoryJob::KomparatorEmptyDirectoryJob( QWidget *_parent )
	: QObject( (QObject*)_parent ), m_parent( _parent )
{
	m_running = 0;
	m_status = i18n( "Searching for empty directories..." );
	m_progress = 0;
	m_total = 0;

	connect( this, SIGNAL( emitEmptyFinished() ), (QObject *)m_parent, SLOT( slotInteractionEmptyFinished() ) );
	connect( this, SIGNAL( emitInteractionMessage( const QString &, const QString & ) ), (QObject *)m_parent, SLOT( slotInteractionMessageError( const QString &, const QString & ) ) );
	connect( this, SIGNAL( emitInteractionProgress( const QString &, const int & ) ), (QObject *)m_parent, SLOT( slotInteractionProgressProgress( const QString &, const int & ) ) );
}

KomparatorEmptyDirectoryJob::~KomparatorEmptyDirectoryJob()
{
}

bool KomparatorEmptyDirectoryJob::initialize( KUrl::List *url_list, KUrl::List *file_list )
{
	m_url_list.clear();
	m_search_list.clear();
	m_empty_directory_list.clear();

	m_progress = 0;

	m_url_list = *url_list;
	qSort( m_url_list );
	m_file_list = *file_list;
	qSort( m_file_list );

	m_total = m_url_list.count();

	m_initialized = true;
	m_canceled = false;

	return true;
}

void KomparatorEmptyDirectoryJob::run()
{
	qRegisterMetaType<KIO::filesize_t>( "KIO::filesize_t" );
	qRegisterMetaType<KIO::UDSEntryList>( "KIO::UDSEntryList" );

	if ( !m_initialized ) return;

	// use m_file_list to sort out some folders
	KUrl::List new_url_list;

	KUrl::List::iterator url_it;
	KUrl::List::iterator file_it=m_file_list.begin();
	for ( url_it=m_url_list.begin(); url_it != m_url_list.end(); ++url_it )
	{
		bool skip = false;
		for ( ; file_it != m_file_list.end(); ++file_it )
		{
			if ( (*url_it).upUrl().isParentOf( (*file_it) ) )
			{
				skip = true;
				break;
			}
		}
		if ( !skip )
		{
			new_url_list.push_back( (*url_it) );
		}
	}
	m_url_list = new_url_list;
	m_total = m_url_list.count();

	if ( m_url_list.count() == 0 )
	{
		emitProgress( m_status, -1 );

		if ( !isCanceled() )
		{
			emit emitEmptyFinished();
		}
		return;
	}

	bool skip;
	KUrl::List appended_urls;

	// Start from end of the list, because parent directories usually come first in the list
	// (starting from the beginning, we would have to search parent directories AND child directories).
	KUrl::List::iterator it = m_url_list.end();
	for ( --it; ; --it )
	{
		if ( isCanceled() ) break;

		skip = false;

		// skip already appended Urls
		KUrl::List::iterator it2;
		for ( it2=appended_urls.begin(); it2 != appended_urls.end(); ++it2 )
		{
			if ( (*it2).equals( (*it).upUrl(), KUrl::CompareWithoutTrailingSlash ) ) skip = true;
			if ( (*it).upUrl().isParentOf( *it2 ) ) skip = true;
		}

		if ( !skip )
		{
			m_search_list.append( (*it).upUrl() );
		}
		else
		{
			incrementProgress();
		}

		appended_urls.push_back( (*it).upUrl() );

		if ( it == m_url_list.begin() ) break;
	}

	if ( m_search_list.count() == 0 )
	{
		emitProgress( m_status, -1 );

		if ( !isCanceled() )
		{
			emit emitEmptyFinished();
		}
		m_initialized = false;
		return;
	}

	KIO::Job *job = KIO::listDir( m_search_list.last(), KIO::HideProgressInfo /*progress*/, true /*hidden m_files*/ );
	connect( job, SIGNAL( entries( KIO::Job *, const KIO::UDSEntryList & ) ), (QObject*)this, SLOT( slotFile( KIO::Job *, const KIO::UDSEntryList & ) ) );
	connect( job, SIGNAL( result( KJob * ) ), (QObject*)this, SLOT( slotFinished( KJob * ) ) );
	m_search_list.pop_back();
}

void KomparatorEmptyDirectoryJob::slotFile( KIO::Job *job, const KIO::UDSEntryList &list )
{
	KIO::UDSEntryList::ConstIterator end = list.end();
	int i;
	for ( i=0; i<list.size(); i++ /*KIO::UDSEntryList::ConstIterator it = list.begin(); it != end; ++it*/ )
	{
		KFileItem tmp_item = KFileItem( KIO::UDSEntry( list.at( i ) /**it*/ ), ((KIO::ListJob *)job)->url(), true, true );
		if ( tmp_item.url().fileName() != ((KIO::ListJob *)job)->url().fileName() && tmp_item.url().fileName() != ".." ) // There is something in the directory
		{
			disconnect( job, SIGNAL( result( KJob * ) ), this, SLOT( slotFinished( KJob * ) ) );
			job->kill( KJob::Quietly );

			if ( !m_search_list.isEmpty() )
			{
				KIO::Job *new_job = KIO::listDir( m_search_list.last(), KIO::HideProgressInfo /*progress*/, true /*hidden m_files*/ );
				connect( new_job, SIGNAL( entries( KIO::Job *, const KIO::UDSEntryList & ) ), (QObject*)this, SLOT( slotFile( KIO::Job *, const KIO::UDSEntryList & ) ) );
				connect( new_job, SIGNAL( result( KJob * ) ), (QObject*)this, SLOT( slotFinished( KJob * ) ) );
				m_search_list.pop_back();
			}
			else
			{
				emitProgress( m_status, -1 );

				if ( !isCanceled() )
				{
					emit emitEmptyFinished();
				}

				m_initialized = false;
				return;
			}

			incrementProgress();

			break;
		}
	}
}

void KomparatorEmptyDirectoryJob::slotFinished( KJob *job )
{
	incrementProgress();

	if ( job->error() == 0 )
	{
		m_empty_directory_list.append( ((KIO::SimpleJob*)job)->url() );
	}

	if ( !m_search_list.isEmpty() )
	{
		KIO::Job *new_job = KIO::listDir( m_search_list.last(), KIO::HideProgressInfo /*progress*/, true /*hidden m_files*/ );
		connect( new_job, SIGNAL( entries( KIO::Job *, const KIO::UDSEntryList & ) ), (QObject*)this, SLOT( slotFile( KIO::Job *, const KIO::UDSEntryList & ) ) );
		connect( new_job, SIGNAL( result( KJob * ) ), (QObject*)this, SLOT( slotFinished( KJob * ) ) );
		m_search_list.pop_back();
	}
	else
	{
		emitProgress( m_status, -1 );

		if ( !isCanceled() )
		{
			emit emitEmptyFinished();
		}

		m_initialized = false;
	}
}

const KUrl::List &KomparatorEmptyDirectoryJob::getEmptyDirectoryList()
{
	return m_empty_directory_list;
}

void KomparatorEmptyDirectoryJob::emitProgress( QString str, int percent )
{
	emit emitInteractionProgress( str, percent );
}

void KomparatorEmptyDirectoryJob::emitMessage( const QString &message, const QString &title )
{
	emit emitInteractionMessage( message, title );
}

void KomparatorEmptyDirectoryJob::incrementProgress()
{
	m_progress++;

	if ( !isCanceled() ) emitProgress( m_status, (int)((m_progress*100)/m_total) );
}

void KomparatorEmptyDirectoryJob::start()
{
	m_canceled = false;
	run();
}

void KomparatorEmptyDirectoryJob::stop()
{
	m_canceled = true;
}
