/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.commands.ReplicableCommand;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.config.RuntimeConfig;
import org.jboss.cache.factories.ComponentRegistry;
import org.jboss.cache.factories.annotations.Inject;
import org.jboss.cache.factories.annotations.Start;
import org.jboss.cache.factories.annotations.Stop;
import org.jboss.cache.interceptors.InterceptorChain;
import org.jboss.cache.invocation.InvocationContextContainer;
import org.jboss.cache.lock.LockManager;
import org.jboss.cache.lock.LockUtil;
import org.jboss.cache.lock.TimeoutException;
import org.jboss.cache.marshall.CommandAwareRpcDispatcher;
import org.jboss.cache.marshall.InactiveRegionAwareRpcDispatcher;
import org.jboss.cache.marshall.Marshaller;
import org.jboss.cache.notifications.Notifier;
import org.jboss.cache.remoting.jgroups.ChannelMessageListener;
import org.jboss.cache.statetransfer.StateTransferManager;
import org.jboss.cache.transaction.GlobalTransaction;
import org.jboss.cache.transaction.TransactionTable;
import org.jboss.cache.util.ThreadGate;
import org.jboss.cache.util.reflect.ReflectionUtil;
import org.jgroups.Address;
import org.jgroups.Channel;
import org.jgroups.ChannelException;
import org.jgroups.ChannelFactory;
import org.jgroups.ExtendedMembershipListener;
import org.jgroups.JChannel;
import org.jgroups.StateTransferException;
import org.jgroups.View;
import org.jgroups.blocks.GroupRequest;
import org.jgroups.blocks.RspFilter;
import org.jgroups.util.Rsp;
import org.jgroups.util.RspList;

import javax.transaction.TransactionManager;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.Vector;

/**
 * Manager that handles all RPC calls between JBoss Cache instances
 *
 * @author <a href="mailto:manik@jboss.org">Manik Surtani (manik@jboss.org)</a>
 */
public class RPCManagerImpl implements RPCManager
{
   private Channel channel;
   private final Log log = LogFactory.getLog(RPCManagerImpl.class);
   private List<Address> members;

   private final Object coordinatorLock = new Object();
   /**
    * True if this Cache is the coordinator.
    */
   private volatile boolean coordinator = false;
   /**
    * Thread gate used to block Dispatcher during JGroups FLUSH protocol
    */
   private final ThreadGate flushBlockGate = new ThreadGate();
   /**
    * JGroups RpcDispatcher in use.
    */
   private CommandAwareRpcDispatcher rpcDispatcher = null;

   /**
    * JGroups message listener.
    */
   private ChannelMessageListener messageListener;
   private Configuration configuration;
   private Notifier notifier;
   private CacheSPI spi;
   private InvocationContextContainer invocationContextContainer;
   private final boolean trace = log.isTraceEnabled();
   private Marshaller marshaller;
   private TransactionManager txManager;
   private TransactionTable txTable;
   private InterceptorChain interceptorChain;

   private boolean isUsingBuddyReplication;
   private boolean isInLocalMode;
   private ComponentRegistry componentRegistry;
   private LockManager lockManager;

   @Inject
   private void setupDependencies(ChannelMessageListener messageListener, Configuration configuration, Notifier notifier,
                                  CacheSPI spi, Marshaller marshaller, TransactionTable txTable,
                                  TransactionManager txManager, InvocationContextContainer container, InterceptorChain interceptorChain,
                                  ComponentRegistry componentRegistry, LockManager lockManager)
   {
      this.messageListener = messageListener;
      this.configuration = configuration;
      this.notifier = notifier;
      this.spi = spi;
      this.marshaller = marshaller;
      this.txManager = txManager;
      this.txTable = txTable;
      this.invocationContextContainer = container;
      this.interceptorChain = interceptorChain;
      this.componentRegistry = componentRegistry;
      this.lockManager = lockManager;
   }

   // ------------ START: Lifecycle methods ------------

   @Start(priority = 15)
   public void start()
   {
      switch (configuration.getCacheMode())
      {
         case LOCAL:
            log.debug("cache mode is local, will not create the channel");
            isInLocalMode = true;
            isUsingBuddyReplication = false;
            break;
         case REPL_SYNC:
         case REPL_ASYNC:
         case INVALIDATION_ASYNC:
         case INVALIDATION_SYNC:
            isInLocalMode = false;
            isUsingBuddyReplication = configuration.getBuddyReplicationConfig() != null && configuration.getBuddyReplicationConfig().isEnabled();
            if (log.isDebugEnabled()) log.debug("Cache mode is " + configuration.getCacheMode());

            boolean fetchState = shouldFetchStateOnStartup();
            initialiseChannelAndRpcDispatcher(fetchState);

            if (fetchState)
            {
               try
               {
                  long start = System.currentTimeMillis();
                  // connect and state transfer
                  channel.connect(configuration.getClusterName(), null, null, configuration.getStateRetrievalTimeout());
                  //if I am not the only and the first member than wait for a state to arrive
                  if (getMembers().size() > 1) messageListener.waitForState();

                  if (log.isDebugEnabled())
                     log.debug("connected, state was retrieved successfully (in " + (System.currentTimeMillis() - start) + " milliseconds)");
               }
               catch (StateTransferException ste)
               {
                  // make sure we disconnect from the channel before we throw this exception!
                  // JBCACHE-761
                  disconnect();
                  throw new CacheException("Unable to fetch state on startup", ste);
               }
               catch (ChannelException e)
               {
                  throw new CacheException("Unable to connect to JGroups channel", e);
               }
               catch (Exception ex)
               {
                  throw new CacheException("Unable to fetch state on startup", ex);
               }
            }
            else
            {
               //otherwise just connect
               try
               {
                  channel.connect(configuration.getClusterName());
               }
               catch (ChannelException e)
               {
                  throw new CacheException("Unable to connect to JGroups channel", e);
               }
            }
            if (log.isInfoEnabled()) log.info("Cache local address is " + getLocalAddress());
      }
   }

   public void disconnect()
   {
      if (channel != null && channel.isOpen())
      {
         log.info("Disconnecting and closing the Channel");
         channel.disconnect();
         channel.close();
      }
   }

   @Stop(priority = 8)
   public void stop()
   {
      try
      {
         disconnect();
      }
      catch (Exception toLog)
      {
         log.error("Problem closing channel; setting it to null", toLog);
      }

      channel = null;
      configuration.getRuntimeConfig().setChannel(null);
      if (rpcDispatcher != null)
      {
         log.info("Stopping the RpcDispatcher");
         rpcDispatcher.stop();
      }

      if (members != null) members = null;

      coordinator = false;

      rpcDispatcher = null;
   }

   /**
    * @return true if we need to fetch state on startup.  I.e., initiate a state transfer.
    */
   private boolean shouldFetchStateOnStartup()
   {
      boolean loaderFetch = configuration.getCacheLoaderConfig() != null && configuration.getCacheLoaderConfig().isFetchPersistentState();
      return !configuration.isInactiveOnStartup() && !isUsingBuddyReplication && (configuration.isFetchInMemoryState() || loaderFetch);
   }

   @SuppressWarnings("deprecation")
   private void initialiseChannelAndRpcDispatcher(boolean fetchState) throws CacheException
   {
      channel = configuration.getRuntimeConfig().getChannel();
      if (channel == null)
      {
         // Try to create a multiplexer channel
         channel = getMultiplexerChannel();

         if (channel != null)
         {
            ReflectionUtil.setValue(configuration, "accessible", true);
            configuration.setUsingMultiplexer(true);
            if (log.isDebugEnabled())
               log.debug("Created Multiplexer Channel for cache cluster " + configuration.getClusterName() + " using stack " + configuration.getMultiplexerStack());
         }
         else
         {
            try
            {
               if (configuration.getClusterConfig() == null)
               {
                  log.debug("setting cluster properties to default value");
                  channel = new JChannel(configuration.getDefaultClusterConfig());
               }
               else
               {
                  if (trace)
                  {
                     log.trace("Cache cluster properties: " + configuration.getClusterConfig());
                  }
                  channel = new JChannel(configuration.getClusterConfig());
               }
            }
            catch (ChannelException e)
            {
               throw new CacheException(e);
            }
         }

         configuration.getRuntimeConfig().setChannel(channel);
      }

      // Channel.LOCAL *must* be set to false so we don't see our own messages - otherwise invalidations targeted at
      // remote instances will be received by self.
      channel.setOpt(Channel.LOCAL, false);
      channel.setOpt(Channel.AUTO_RECONNECT, true);
      channel.setOpt(Channel.AUTO_GETSTATE, fetchState);
      channel.setOpt(Channel.BLOCK, true);

      // always use the InactiveRegionAwareRpcDispatcher - exceptions due to regions not being active should not propagate to remote
      // nodes as errors. - Manik
      // but only if we are using region based marshalling?!??
      if (configuration.isUseRegionBasedMarshalling())
      {
         rpcDispatcher = new InactiveRegionAwareRpcDispatcher(channel, messageListener, new MembershipListenerAdaptor(),
               spi, invocationContextContainer, interceptorChain, componentRegistry);
      }
      else
      {
         rpcDispatcher = new CommandAwareRpcDispatcher(channel, messageListener, new MembershipListenerAdaptor(),
               invocationContextContainer, invocationContextContainer, interceptorChain, componentRegistry);
      }

      rpcDispatcher.setRequestMarshaller(marshaller);
      rpcDispatcher.setResponseMarshaller(marshaller);
   }

   public Channel getChannel()
   {
      return channel;
   }


   private JChannel getMultiplexerChannel() throws CacheException
   {
      String stackName = configuration.getMultiplexerStack();

      RuntimeConfig rtc = configuration.getRuntimeConfig();
      ChannelFactory channelFactory = rtc.getMuxChannelFactory();
      JChannel muxchannel = null;

      if (channelFactory != null)
      {
         try
         {
            muxchannel = (JChannel) channelFactory.createMultiplexerChannel(stackName, configuration.getClusterName());
         }
         catch (Exception e)
         {
            throw new CacheException("Failed to create multiplexed channel using stack " + stackName, e);
         }
      }

      return muxchannel;
   }


   private void removeLocksForDeadMembers(NodeSPI node, List deadMembers)
   {
      Set<GlobalTransaction> deadOwners = new HashSet<GlobalTransaction>();
      Object owner = lockManager.getWriteOwner(node);

      if (isLockOwnerDead(owner, deadMembers)) deadOwners.add((GlobalTransaction) owner);


      for (Object readOwner : lockManager.getReadOwners(node))
      {
         if (isLockOwnerDead(readOwner, deadMembers)) deadOwners.add((GlobalTransaction) readOwner);
      }

      for (GlobalTransaction deadOwner : deadOwners)
      {
         boolean localTx = deadOwner.getAddress().equals(getLocalAddress());
         boolean broken = LockUtil.breakTransactionLock(node, lockManager, deadOwner, localTx, txTable, txManager);

         if (broken && trace) log.trace("Broke lock for node " + node.getFqn() + " held by " + deadOwner);
      }

      // Recursively unlock children
      for (Object child : node.getChildrenDirect())
      {
         removeLocksForDeadMembers((NodeSPI) child, deadMembers);
      }
   }

   private boolean isLockOwnerDead(Object owner, List deadMembers)
   {
      boolean result = false;
      if (owner != null && owner instanceof GlobalTransaction)
      {
         Object addr = ((GlobalTransaction) owner).getAddress();
         result = deadMembers.contains(addr);
      }
      return result;
   }

   // ------------ END: Lifecycle methods ------------

   // ------------ START: RPC call methods ------------

   public List<Object> callRemoteMethods(Vector<Address> recipients, ReplicableCommand command, int mode, long timeout, boolean useOutOfBandMessage) throws Exception
   {
      return callRemoteMethods(recipients, command, mode, timeout, null, useOutOfBandMessage);
   }

   public List<Object> callRemoteMethods(Vector<Address> recipients, ReplicableCommand command, boolean synchronous, long timeout, boolean useOutOfBandMessage) throws Exception
   {
      return callRemoteMethods(recipients, command, synchronous ? GroupRequest.GET_ALL : GroupRequest.GET_NONE, timeout, useOutOfBandMessage);
   }

   public List<Object> callRemoteMethods(Vector<Address> recipients, ReplicableCommand command, int mode, long timeout, RspFilter responseFilter, boolean useOutOfBandMessage) throws Exception
   {
      // short circuit if we don't have an RpcDispatcher!
      if (rpcDispatcher == null) return null;

      int modeToUse = mode;
      int preferredMode;
      if ((preferredMode = spi.getInvocationContext().getOptionOverrides().getGroupRequestMode()) > -1)
         modeToUse = preferredMode;

      if (trace)
         log.trace("callRemoteMethods(): valid members are " + recipients + " methods: " + command + " Using OOB? " + useOutOfBandMessage);

      if (channel.flushSupported() && !flushBlockGate.await(configuration.getStateRetrievalTimeout()))
      {
         throw new TimeoutException("State retrieval timed out waiting for flush unblock.");
      }

      useOutOfBandMessage = false;

      RspList rsps = rpcDispatcher.invokeRemoteCommands(recipients, command, modeToUse, timeout, isUsingBuddyReplication, useOutOfBandMessage, responseFilter);

      if (mode == GroupRequest.GET_NONE) return Collections.emptyList();// async case

      if (trace)
         log.trace("(" + getLocalAddress() + "): responses for method " + command.getClass().getSimpleName() + ":\n" + rsps);

      // short-circuit no-return-value calls.
      if (rsps == null) return Collections.emptyList();

      List<Object> retval = new ArrayList<Object>(rsps.size());

      for (Rsp rsp : rsps.values())
      {
         if (rsp.wasSuspected() || !rsp.wasReceived())
         {
            CacheException ex;
            if (rsp.wasSuspected())
            {
               ex = new SuspectException("Suspected member: " + rsp.getSender());
            }
            else
            {
               ex = new TimeoutException("Replication timeout for " + rsp.getSender());
            }
            retval.add(new ReplicationException("rsp=" + rsp, ex));
         }
         else
         {
            Object value = rsp.getValue();
            if (value instanceof Exception && !(value instanceof ReplicationException))
            {
               // if we have any application-level exceptions make sure we throw them!!
               if (trace) log.trace("Recieved exception'" + value + "' from " + rsp.getSender());
               throw (Exception) value;
            }
            retval.add(value);
         }
      }
      return retval;
   }

   // ------------ START: Partial state transfer methods ------------

   public void fetchPartialState(List<Address> sources, Fqn sourceTarget, Fqn integrationTarget) throws Exception
   {
      String encodedStateId = sourceTarget + StateTransferManager.PARTIAL_STATE_DELIMITER + integrationTarget;
      fetchPartialState(sources, encodedStateId);
   }

   public void fetchPartialState(List<Address> sources, Fqn subtree) throws Exception
   {
      if (subtree == null)
      {
         throw new IllegalArgumentException("Cannot fetch partial state. Null subtree.");
      }
      fetchPartialState(sources, subtree.toString());
   }

   private void fetchPartialState(List<Address> sources, String stateId) throws Exception
   {
      if (sources == null || sources.isEmpty() || stateId == null)
      {
         // should this really be throwing an exception?  Are there valid use cases where partial state may not be available? - Manik
         // Yes -- cache is configured LOCAL but app doesn't know it -- Brian
         //throw new IllegalArgumentException("Cannot fetch partial state, targets are " + sources + " and stateId is " + stateId);
         if (log.isWarnEnabled())
            log.warn("Cannot fetch partial state, targets are " + sources + " and stateId is " + stateId);
         return;
      }

      List<Address> targets = new LinkedList<Address>(sources);

      //skip *this* node as a target
      targets.remove(getLocalAddress());

      if (targets.isEmpty())
      {
         // Definitely no exception here -- this happens every time the 1st node in the
         // cluster activates a region!! -- Brian
         if (log.isDebugEnabled()) log.debug("Cannot fetch partial state. There are no target members specified");
         return;
      }

      if (log.isDebugEnabled())
         log.debug("Node " + getLocalAddress() + " fetching partial state " + stateId + " from members " + targets);
      boolean successfulTransfer = false;
      for (Address target : targets)
      {
         if (log.isDebugEnabled())
            log.debug("Node " + getLocalAddress() + " fetching partial state " + stateId + " from member " + target);
         messageListener.setStateSet(false);
         successfulTransfer = channel.getState(target, stateId, configuration.getStateRetrievalTimeout());
         if (successfulTransfer)
         {
            try
            {
               messageListener.waitForState();
            }
            catch (Exception transferFailed)
            {
               successfulTransfer = false;
            }
         }
         if (log.isDebugEnabled())
            log.debug("Node " + getLocalAddress() + " fetching partial state " + stateId + " from member " + target + (successfulTransfer ? " successful" : " failed"));
         if (successfulTransfer) break;
      }

      if (!successfulTransfer)
      {
         if (log.isDebugEnabled())
            log.debug("Node " + getLocalAddress() + " could not fetch partial state " + stateId + " from any member " + targets);
      }
   }

   // ------------ END: Partial state transfer methods ------------

   // ------------ START: Informational methods ------------

   public Address getLocalAddress()
   {
      return channel != null ? channel.getLocalAddress() : null;
   }

   public List<Address> getMembers()
   {
      if (isInLocalMode) return null;
      if (members == null)
         return Collections.emptyList();
      else
         return members;
   }

   public boolean isCoordinator()
   {
      return coordinator;
   }

   public Address getCoordinator()
   {
      if (channel == null)
      {
         return null;
      }

      synchronized (coordinatorLock)
      {
         while (members == null || members.isEmpty())
         {
            log.debug("getCoordinator(): waiting on viewAccepted()");
            try
            {
               coordinatorLock.wait();
            }
            catch (InterruptedException e)
            {
               log.error("getCoordinator(): Interrupted while waiting for members to be set", e);
               break;
            }
         }
         return members != null && members.size() > 0 ? members.get(0) : null;
      }
   }

   // ------------ END: Informational methods ------------

   /*----------------------- MembershipListener ------------------------*/

   protected class MembershipListenerAdaptor implements ExtendedMembershipListener
   {

      public void viewAccepted(View newView)
      {
         Vector<Address> newMembers = newView.getMembers();
         if (log.isInfoEnabled()) log.info("Received new cluster view: " + newView);
         synchronized (coordinatorLock)
         {
            boolean needNotification = false;
            if (newMembers != null)
            {
               if (members != null)
               {
                  // we had a membership list before this event.  Check to make sure we haven't lost any members,
                  // and if so, determine what members have been removed
                  // and roll back any tx and break any locks
                  List<Address> removed = new ArrayList<Address>(members);
                  removed.removeAll(newMembers);
                  removeLocksForDeadMembers(spi.getRoot(), removed);
               }

               members = new ArrayList<Address>(newMembers); // defensive copy.

               needNotification = true;
            }

            // Now that we have a view, figure out if we are the coordinator
            coordinator = (members != null && members.size() != 0 && members.get(0).equals(getLocalAddress()));

            // now notify listeners - *after* updating the coordinator. - JBCACHE-662
            if (needNotification && notifier != null)
            {
               InvocationContext ctx = spi.getInvocationContext();
               notifier.notifyViewChange(newView, ctx);
            }

            // Wake up any threads that are waiting to know about who the coordinator is
            coordinatorLock.notifyAll();
         }
      }

      /**
       * Called when a member is suspected.
       */
      public void suspect(Address suspected_mbr)
      {
      }

      /**
       * Indicates that a channel has received a BLOCK event from FLUSH protocol.
       */
      public void block()
      {
         flushBlockGate.close();
         if (log.isDebugEnabled()) log.debug("Block received at " + getLocalAddress());
         notifier.notifyCacheBlocked(true);
         notifier.notifyCacheBlocked(false);

         if (log.isDebugEnabled()) log.debug("Block processed at " + getLocalAddress());
      }

      /**
       * Indicates that a channel has received a UNBLOCK event from FLUSH protocol.
       */
      public void unblock()
      {
         if (log.isDebugEnabled()) log.debug("UnBlock received at " + getLocalAddress());

         notifier.notifyCacheUnblocked(true);
         notifier.notifyCacheUnblocked(false);

         if (log.isDebugEnabled()) log.debug("UnBlock processed at " + getLocalAddress());
         flushBlockGate.open();
      }

   }
   /*------------------- End of MembershipListener ----------------------*/
}