/*################################################################################
# Linux Management Providers (LMP), Provider Common Library
# Copyright (C) 2007 Frederic Desmons, ETRI <desmons@etri.re.kr, desmons_frederic@yahoo.fr>
#
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
#
# Modified by 2008 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr, guillaumebottex@gmail.com>
################################################################################*/

#include "Common.h"
#include "net_dev.h"
#include <netdb.h>
#include <sys/utsname.h>

using namespace std;

/* ---------------------------------------------------------------------------*/
/*                                System Calls                                */
/* ---------------------------------------------------------------------------*/

int CF_runCommand(const string& cmd, string& stdout_msg, string& stderr_msg, string& errorMessage) {
	CF_assert(CF_runCommand(cmd, stdout_msg, stderr_msg, false, errorMessage));
	return OK;
}

// Modified by Ilsoo Byun (2007/08/14)
//		- Removal of disk access
//		- Sync problem is also solved.
// Modified by Frederic Desmons (2007/06/15)
// Multi-Process support for runCommand
int CF_runCommand(const string& cmd, string& stdout_msg, string& stderr_msg, bool do_trace, string& errorMessage) {

	stdout_msg.clear();
	stderr_msg.clear();

	// to get stdout
	int stdout_fd[2];
	if (pipe(stdout_fd) == -1) {
		errorMessage = strerror(errno);
		return FAILED;
	}

	// to get stderr
	int stderr_fd[2];
	if (pipe(stderr_fd) == -1) {
		errorMessage = strerror(errno);
		return FAILED;
	}

	pid_t pid = fork();
	if (pid == -1) {
		errorMessage = strerror(errno);
		return FAILED;
	}

	if (pid != 0) {	// parent process
		// close the write side
		if (close(stdout_fd[1]) < 0) {
			errorMessage = strerror(errno);
			return FAILED;
		}
		if (close(stderr_fd[1]) < 0) {
			errorMessage = strerror(errno);
			return FAILED;
		}

		fd_set readfds;

		int fdmax;
		if (stdout_fd[0] > stderr_fd[0])
			fdmax = stdout_fd[0];
		else
			fdmax = stderr_fd[0];

		char buf[513];

		bool stdout_done = false;
		bool stderr_done = false;

		while (1) {

			if (stdout_done && stderr_done)
				break;

			FD_ZERO(&readfds);
			if (!stdout_done)
				FD_SET(stdout_fd[0], &readfds);
			if (!stderr_done)
				FD_SET(stderr_fd[0], &readfds);

			if (select(fdmax+1, &readfds, NULL, NULL, NULL) < 0) {
				errorMessage = strerror(errno);
				return FAILED;
			}

			if (FD_ISSET(stdout_fd[0], &readfds)) {
				ssize_t count = read(stdout_fd[0], buf, 512);
				if (count == 0) { // EOF
					stdout_done = true;
					if (close(stdout_fd[0]) < 0) {
						errorMessage = strerror(errno);
						return FAILED;
					}
					continue;
				}
				if (count < 0) {
					errorMessage = strerror(errno);
					return FAILED;
				}
				buf[count] = '\0';
				// success
				stdout_msg.append(buf);
				continue;
			}

			if (FD_ISSET(stderr_fd[0], &readfds)) {
				ssize_t count = read(stderr_fd[0], buf, 512);
				if (count == 0) { // EOF
					stderr_done = true;
					if (close(stderr_fd[0]) < 0) {
						errorMessage = strerror(errno);
						return FAILED;
					}
					continue;
				}
				if (count < 0) {
					errorMessage = strerror(errno);
					return FAILED;
				}
				buf[count] = '\0';
				// success
				stderr_msg.append(buf);
				// print strace result into the debug file
				// warning: strace result is added to stderr
				if (do_trace)
					system(("echo \"" + (string) buf + "\" >> cmpi_prov_debug.txt").c_str());
				continue;
			}

		}

		int status;
		waitpid (pid, &status, 0);

		if (status == -1 || status == 127) {
			errorMessage = "Failed to execute: "+cmd+"\n"+stderr_msg;
			return FAILED;
		}

		return OK;

	} else {	// child process
		// close the read side
		if (close(stdout_fd[0]) < 0) {
			errorMessage = strerror(errno);
			exit(FAILED);
		}
		if (close(stderr_fd[0]) < 0) {
			errorMessage = strerror(errno);
			exit(FAILED);
		}

		// redirect stdout
		if (dup2(stdout_fd[1], 1) < 0) {
			errorMessage = strerror(errno);
			exit(FAILED);
		}

		// redirect stderr
		if (dup2(stderr_fd[1], 2) < 0) {
			errorMessage = strerror(errno);
			exit(FAILED);
		}

		// close the write side, this will only be effective when the child fork by 'system' dies
		if (close(stdout_fd[1]) < 0) {
			errorMessage = strerror(errno);
			exit(FAILED);
		}
		if (close(stderr_fd[1]) < 0) {
			errorMessage = strerror(errno);
			exit(FAILED);
		}

		// openwbem sets the sigmask so the executed command may block.
		// to avoid this situation we set to an empty mask.
		sigset_t set, old_set;
		sigemptyset(&set);
		sigprocmask(SIG_SETMASK, &set, &old_set);

		// in standard English please ;)
		string _cmd = cmd;

		if (do_trace)
			_cmd = "/usr/bin/strace " + _cmd;

		_cmd = "LANG=C " + _cmd;

		// execute the command
		int rc = system(_cmd.c_str());

		if (rc != 0)
			exit(FAILED);

		exit(OK);

	}

}

// Added by Frederic Desmons (2007/07/10)
int CF_runCommandFL(const string& cmd, string& stdOut, string& errorMessage) {
	string stdErr;
	CF_assert(CF_runCommand(cmd, stdOut, stdErr, errorMessage));
	size_t i = stdOut.find('\n');
	if (i != string::npos)
		stdOut.erase(i, string::npos);
	return OK;
}

// Added by Frederic Desmons (2007/07/16)
int CF_runCommandToLines(const string& cmd, vector<string>& output, unsigned long expected_number_of_lines, string& errorMessage) {
  	string cmd_output, stdErr;
  	CF_assert(CF_runCommand(cmd, cmd_output, stdErr, errorMessage));
  	output.clear();
  	CF_splitText(output, cmd_output, '\n');
  	if (expected_number_of_lines != 0 && output.size() != expected_number_of_lines) {
  		errorMessage = "Wrong format: " + cmd + "output";
  		return FAILED;
  	}
  	return OK;
}

// Modified by Ilsoo Byun (2007/05/09)
// If '/bin/hostname -f' fails, it tries to execute '/bin/hostname' without options.
int CF_getSystemName(string& sysName, string& errorMessage) {
/*	string value, errout;
	if (CF_runCommand("/bin/hostname -f", sysName, errout, errorMessage)!=OK) {
		CF_assert( CF_runCommand("/bin/hostname", sysName, errout, errorMessage));
	}
	sysName = CF_trimText(sysName);
	return OK;*/
	/*string ip;
	CF_getMachineIP(ip, errorMessage);
	struct hostent* host = gethostbyname(ip.c_str());
	if (host == NULL)
		return FAILED;

	sysName = host->h_name;
	free(host);
	return OK;*/

	// Modified by Ilsoo Byun(2008/10/02)
	struct utsname uts;
        if (uname(&uts) < 0) {
                errorMessage = "Failed to get system name: " + (string) strerror(errno);
                return FAILED;
        }

        sysName = uts.nodename;
	return OK;
}

// Added by Frederic Desmons (2007/06/28)
// Modified by Frederic Desmons (2007/07/10)
int CF_getOSName(string& OSName, string& errorMessage) {
	//CF_assert(CF_runCommandFL("/bin/uname -o", OSName, errorMessage));
	//return OK;

	struct utsname uts;
        if (uname(&uts) < 0) {
                errorMessage = "failed to get os: " + (string) strerror(errno);
                return FAILED;
        }

        OSName = uts.sysname;
	return OK;
}

// Added by Ilsoo Byun (2007/05/09)
int CF_getEthernetPortNames(vector<string>& result, string& errorMessage) {

	// Modified by Guillaume BOTTEX (2008/11/21)
	// Now using sockets to get the Ethernet port names
	struct net_proc* devices;
	int count = CN_getNetDevices(&devices);

	for (int i = 0; i < count; ++i)
	{
		if (!CF_startsWith(devices[i].name, "eth"))
			continue;

		result.push_back(devices[i].name);

	}
	CN_releaseNetDevices(&devices, count);

	sort(result.begin(), result.end());

	return OK;
}

// Added by Ilsoo Byun (2007/05/09)
int CF_getIP(const string& ethName, string& ip, string& errorMessage) {
	// Modified by Frederic Desmons (2007/11/12)
	// Get only the first line of the std output
	/*string value;
	CF_assert( CF_runCommandFL("/sbin/ifconfig "+ethName+" | awk '/HWaddr/{} /inet addr/{print $2}'", value, errorMessage) );
	ip = CF_trimText(value);
	if ( CF_startsWithNoCase(ip, "addr:") )
		ip = ip.substr(5);
	return OK;*/

	// Modified by Ilsoo Byun(2008/10/02)
	const char* result = CN_getIP(ethName.c_str());
	if (result == NULL) {
		errorMessage = "CF_getIP failed";
		return FAILED;
	}
	ip = string(result);
	free((void *) result);
	return OK;
}

// Added by Ilsoo Byun (2007/05/09)
int CF_getMachineIP(string& ip, string& errorMessage) {

	//string _ip;
	// Modified by Frederic Desmons (2007/11/12)
	// Ignore loopback and empty results as well (case there is more than one interface)
/*	vector<string> result;
	CF_assert(CF_getEthernetPortNames(result, errorMessage));
	for (unsigned int i=0; i < result.size(); i++) {
		CF_assert(CF_getIP(result[i], _ip, errorMessage));
		if (_ip != "127.0.0.1" && _ip !="") {
			ip = _ip;
			return OK;
		}
	}*/
	// Means the system has no configured interface... unlikely
	//ip = "127.0.0.1";
	//return OK;
	/*
	int rc = OK;
	if (_ip.length() == 0) {
		vector<string> result;
		rc = CF_getEthernetPortNames(result, errorMessage);
		if (rc != OK || result.size() < 1) return rc;
		CF_assert( CF_getIP(result[0], _ip, errorMessage) );
		if (_ip == "127.0.0.1") {
			if (result.size() > 1) {
				CF_assert( CF_getIP(result[1], _ip, errorMessage) );
			} else {
				errorMessage = "Failed to get an IP address";
				return FAILED;
			}
		}
	}
	ip = _ip;
	return rc;
	*/

        struct net_proc* devices;
        int count = CN_getNetDevices(&devices);

        for (int i = 0; i < count; ++i) {
		if (!CF_startsWith(devices[i].name, "eth")) continue;
		const char* _ip = CN_getIP(devices[i].name);
		if (_ip == NULL || strcmp(_ip, "127.0.0.1") == 0) {
			free((void *) _ip);
			continue;
		} else {
			ip = _ip;
			free((void *) _ip);
			return OK;
		}
        }
        CN_releaseNetDevices(&devices, count);

	ip = "127.0.0.1";
	return OK;

}

// Refer to the below URL for block size calculation
// http://www.faqs.org/docs/Linux-mini/Partition.html#BLOCKSIZE
int CF_getBlockSize(const string& path, int& blocksize, int& num_blocks, string& errorMessage) {
	string stdout, stderr, line;
	CF_assert(CF_runCommand("fdisk -l "+path, stdout, stderr, errorMessage));
	istringstream iss(stdout);
	int heads = 0;
	int sectors = 0;
	int bytes = 0;
	unsigned long long p_size = 0;
	while (getline(iss, line)) {
		if (CF_startsWith(line, "Disk "+path)) {
			vector<string> tokens;
			CF_splitTextBySpace(tokens, line);
			p_size = atoi(tokens[4].c_str());
		}

		if (heads == 0) {
			string::size_type pos = line.find("heads");
			if (pos != string::npos) {
				string heads_str = line.substr(0, pos);
				heads = atoi(heads_str.c_str());
			}
		}

		if (sectors == 0) {
			string::size_type pos = line.find(" sectors/track");
			if (pos != string::npos) {
				string sectors_str = line.substr(0, pos);
				sectors_str = sectors_str.substr(sectors_str.find_last_of(' '));
				sectors = atoi(sectors_str.c_str());
			}
		}

		if (bytes == 0) {
			const string start_line = "Units = cylinders of";
			if (CF_startsWith(line, start_line)) {
				string::size_type pos = line.find("*");
				bytes = atoi(line.substr(pos+2).c_str());

			}
		}
	}
	blocksize = heads * sectors * bytes / 1024;
	if (blocksize == 0) {
		errorMessage = "Failed to calculate the block size of "+path;
		return FAILED;
	}
	num_blocks = p_size / blocksize;
	return OK;
}

int CF_getWhatisFL(const string& name, string& output, string& errorMessage)
{
	CF_assert(CF_runCommandFL("whatis "+name,output,errorMessage));
	
	int pos;
	
	if((pos=output.find_first_of('-'))!=string::npos)
		output=output.substr(pos+2);
	else
		output.clear();
	
	return OK;
}

/* ---------------------------------------------------------------------------*/
/*                             File manipulation                              */
/* ---------------------------------------------------------------------------*/

extern int errno;
// Added by Ilsoo Byun (2007/06/15)
int CF_lock(const struct flock& file_lock, int& fd, string& errorMessage) {
	_E_;
	const char* lock_file = "/root/.opendrim_lock";
	fd = open(lock_file, O_RDWR | O_CREAT);
	if (fcntl(fd, F_SETLKW, &file_lock) == -1) {
		close(fd);
		string strerr = strerror(errno);
		errorMessage = "Failed to lock: "+strerr;
		return FAILED;
	};
	_L_;
	return OK;
}

// Added by Ilsoo Byun (2007/06/15)
int CF_unlock(struct flock& file_lock, int fd, string& errorMessage) {
	_E_;
	file_lock.l_type = F_UNLCK;
	if (fcntl(fd, F_SETLKW, &file_lock) == -1) {
		close(fd);
		string strerr = strerror(errno);
		errorMessage = "Failed to lock: "+strerr;
		return FAILED;
	}
	close(fd);
	_L_;
	return OK;
}

int CF_readTextFile(const string& path, string& content, string& errorMessage) {
  	FILE* file;
  	char buf[1025];
  	content.clear();
  	file = fopen(path.c_str(), "r");
  	if (file == NULL) {
  		errorMessage = "Unable to open file: " + path;
    	return FAILED;
  	}
  	while(!feof(file)) {
		if (fgets(buf,1024,file) != NULL)
			content += buf;
		else
			break;
  	}
  	fclose(file);
  	return OK;
}

// Added by Frederic Desmons (2007/07/16)
int CF_readTextFileFL(const string& path, string& content, string& errorMessage) {
	vector<string> file_content_lines;
  	CF_assert(CF_readTextFileToLines(path, file_content_lines, 0, errorMessage));
  	if (file_content_lines.size() < 1) {
  		errorMessage = "Wrong format: " + path;
  		return FAILED;
  	}
  	content = file_content_lines[0];
  	return OK;
}

// Added by Frederic Desmons (2007/07/16)
int CF_readTextFileToLines(const string& path, vector<string>& content, unsigned long expected_number_of_lines, string& errorMessage) {
  	string file_content;
  	CF_assert(CF_readTextFile(path, file_content, errorMessage));
  	content.clear();
  	CF_splitText(content, file_content,'\n');
  	if (expected_number_of_lines != 0 && content.size() != expected_number_of_lines) {
  		errorMessage = "Wrong format: " + path;
  		return FAILED;
  	}
  	return OK;
}

int CF_readFileToString(const string& path, string& result, string& errorMessage) {
	ifstream ifs(path.c_str(), ifstream::in);
	if (!ifs.is_open()) {
		errorMessage = "Failed to open: "+path;
		return FAILED;
	}

	string str;
	bool isFirst = true;
	while(getline(ifs, str)) {
		if (isFirst)
			isFirst = false;
		else
			result += "\n";		//new line

		result += str;
	}
	ifs.close();
	return OK;
}

int CF_writeTextFile(const string& path, const string& content, string& errorMessage) {
  	FILE* file;
  	file = fopen(path.c_str(), "w");
  	if (file == NULL) {
  		errorMessage = "Unable to open file: " + path;
    	return FAILED;
  	}
	fputs(content.c_str(), file);
  	fclose(file);
  	return OK;
}

// Added by Ilsoo Byun (2007/05/10)
bool CF_isExist(const string& filePath) {
	FILE * pFile;
	pFile = fopen (filePath.c_str(),"r");
	if (pFile == NULL)
		return false;
	// Added by Frederic Desmons (2007/08/08)
	// Nasty bug, files MUST be closed otherwise re-opening them may not work
	fclose(pFile);
	return true;
}

// Added by Frederic Desmons (2007/08/16)
bool CF_isRegularFile(const string& path) {
	struct stat file_stats;
	if (stat(path.c_str(), &file_stats) != 0)
		return FAILED;
	return (S_ISREG(file_stats.st_mode));
}

// Added by Frederic Desmons (2007/08/16)
bool CF_isDirectory(const string& path) {
	struct stat file_stats;
	if (stat(path.c_str(), &file_stats) != 0)
		return FAILED;
	return (S_ISDIR(file_stats.st_mode));
}

// Added by Frederic Desmons (2007/07/26)
int CF_scanDirectory(const string& directory, vector<string>& regular_files, vector<string>& directories, vector<string>& block_devices, vector<string>& character_devices, string& errorMessage) {
	regular_files.clear();
	directories.clear();
	block_devices.clear();
	character_devices.clear();
	struct dirent **entries;
	int nb_entries = scandir(directory.c_str(), &entries, 0, alphasort);
	if (nb_entries < 0) {
		errorMessage = "Failed to list files: " + directory;
		return FAILED;
	}
	for (int i=0; i < nb_entries; i++) {
		if (((string) entries[i]->d_name) == "." || ((string) entries[i]->d_name) == "..")
			continue;
		switch (entries[i]->d_type) {
			case DT_REG:
				regular_files.push_back(entries[i]->d_name);
				break;
			case DT_DIR:
				directories.push_back(entries[i]->d_name);
				break;
			case DT_BLK:
				block_devices.push_back(entries[i]->d_name);
				break;
			case DT_CHR:
				character_devices.push_back(entries[i]->d_name);
				break;
		}
	}
	return OK;
}


/* ---------------------------------------------------------------------------*/
/*                            String manipulation                             */
/* ---------------------------------------------------------------------------*/

// Added by Guillaume Bottex (2009/04/28)
string CF_removeChar(const string& input, const char* characters)
{
	string output="";
	char* token;
	char* _input;

	_input=(char*)malloc(sizeof(char)*(strlen(input.c_str())+1));
	strcpy(_input,input.c_str());

	token = strtok(_input,characters);
	while (token != NULL)
	{
		output+=(token);
		token = strtok (NULL,characters);
	}

	free(_input);

	return output;
}

// Added by Guillaume Bottex (2008/08/07)
void CF_splitText(vector<string>& output, const string& input, const char* separators)
{
	char* token;
	char* _input;

	output.clear();

	_input=(char*)malloc(sizeof(char)*(strlen(input.c_str())+1));
	strcpy(_input,input.c_str());

	token = strtok(_input,separators);
	while (token != NULL)
	{
		output.push_back(token);
		token = strtok (NULL,separators);
	}

	free(_input);
}

void CF_splitText(vector<string>& output, const string& input, char separator) {
	string buf = input;
	output.clear();
	while (buf.size() > 0) {
		size_t i = buf.find(separator);
		output.push_back(buf.substr(0, i));
		if (i == string::npos)
			break;
		buf.erase(0, i+1);
		if (buf.size()==0)
			output.push_back("");
	}
}

// Modified by Ilsoo Byun (2007/10/24)
//		-To process space characters like ' ', '\t' and '\n'.
void CF_splitTextBySpace(vector<string>& output, const string& input) {
	string buf = input;
	output.clear();
	buf = CF_trimText(buf);
	if (buf.size() == 0) return;

	const char* spaceChr = " \t\r\n";
	size_t i = 0;
	while(i != string::npos) {
		i = buf.find_first_of(spaceChr);

		//If it is a empty string, skip it.
		string parsed = buf.substr(0, i);
		buf.erase(0, i+1);
		parsed = CF_trimText(parsed);
		if (parsed.size() == 0) continue;

		output.push_back(parsed);
	}
}

// Modified by Ilsoo Byun (2007/04/30)
string CF_trimText(const string& text) {
	string str = text;
	const char* spaceChr = " \t\r\n";
	size_t pos = str.find_last_not_of(spaceChr);
	if (pos != string::npos) {
		str.erase(pos+1);
		pos = str.find_first_not_of(spaceChr);
		if (pos != string::npos) str.erase(0, pos);
	} else {
		str.erase(str.begin(), str.end());
	}
	return str;
}

//Added by Frederic Desmons, ETRI (2007/08/23) <desmons@etri.re.kr, desmons_frederic@yahoo.fr>
string CF_untab(const string& text) {
	string str = text;
	for (unsigned int i = 0; i < str.size(); i++) {
		if (str[i] == '\t' || str[i] == '\r')
			str[i] = ' ';
	}
	return str;
}

string CF_toLowCase(const string& str) {
	string result = str;
	for (size_t i=0; i<result.size(); i++) {
		if (result[i] > 64 && result[i] < 91)
			result[i]+=32;
	}
	return result;
}

// Added by Ilsoo Byun (2007/04/30)
// Modifed by Frederic Desmons (2007/05/07): Use of "string" instead of "char*"
bool CF_startsWith(const string& str, const string& word) {
	if (str.size() >= word.size() && str.substr(0, word.size())==word)
		return true;
	return false;
}

// Added by Frederic Desmons (2007/05/07)
bool CF_startsWithNoCase(const string& str, const string& word) {
	string str_low_case=CF_toLowCase(str);
	string word_low_case=CF_toLowCase(word);
	return CF_startsWith(str_low_case, word_low_case);
}

// Added by Ilsoo Byun (2007/04/30)
// Modifed by Frederic Desmons (2007/05/07): Use of "string" instead of "char*"
bool CF_endsWith(const string& str, const string& word) {
	if (str.size() >= word.size()) {
		if (str.substr(str.size()-word.size(), string::npos)==word)
			return true;
		else
			return false;
	}
	return false;
}

// Added by Frederic Desmons (2007/05/07)
bool CF_endsWithNoCase(const string& str, const string& word) {
	string str_low_case=CF_toLowCase(str);
	string word_low_case=CF_toLowCase(word);
	return CF_endsWith(str_low_case, word_low_case);
}

string CF_quoteString(const string& str) {
	string quoted_string;
	vector<string> _str;
	CF_splitText(_str, str, '"');
	for (size_t i = 0; i < _str.size(); i++) {
		quoted_string += _str[i];
		if (i < _str.size()-1)
			quoted_string += "\\\"";
	}
	return quoted_string;
}


/* ---------------------------------------------------------------------------*/
/*                            C++ type comparison                             */
/* ---------------------------------------------------------------------------*/

bool CF_strCmpNoCase(const string& str1, const string& str2) {
	string temp1 = CF_toLowCase(str1);
	string temp2 = CF_toLowCase(str2);
	return temp1==temp2;
}

// Added by Frederic Desmons (2007/07/31)
bool CF_strArrayCmpNoCase(const vector<string>& vect1, const vector<string>& vect2) {
	_ARRAY_CMP
}

// Added by Frederic Desmons (2007/07/31)
bool CF_numArrayCmp(const vector<unsigned char>& vect1, const vector<unsigned char>& vect2) {
	_ARRAY_CMP
}
bool CF_numArrayCmp(const vector<unsigned short>& vect1, const vector<unsigned short>& vect2) {
	_ARRAY_CMP
}
bool CF_numArrayCmp(const vector<unsigned long>& vect1, const vector<unsigned long>& vect2) {
	_ARRAY_CMP
}
bool CF_numArrayCmp(const vector<unsigned long long>& vect1, const vector<unsigned long long>& vect2) {
	_ARRAY_CMP
}
bool CF_numArrayCmp(const vector<signed char>& vect1, const vector<signed char>& vect2) {
	_ARRAY_CMP
}
bool CF_numArrayCmp(const vector<short>& vect1, const vector<short>& vect2) {
	_ARRAY_CMP
}
bool CF_numArrayCmp(const vector<long>& vect1, const vector<long>& vect2) {
	_ARRAY_CMP
}
bool CF_numArrayCmp(const vector<long long>& vect1, const vector<long long>& vect2) {
	_ARRAY_CMP
}
bool CF_numArrayCmp(const vector<float>& vect1, const vector<float>& vect2) {
	_ARRAY_CMP
}
bool CF_numArrayCmp(const vector<double>& vect1, const vector<double>& vect2) {
	_ARRAY_CMP
}

// Added by Frederic Desmons (2007/07/31)
bool CF_boolArrayCmp(const vector<bool>& vect1, const vector<bool>& vect2) {
	_ARRAY_CMP
}

// Added by Frederic Desmons (2007/07/31)
bool CF_refArrayCmp(const vector<Objectpath>& vect1, const vector<Objectpath>& vect2) {
	if (vect1.size() != vect2.size())
		return false;
	for (size_t i=0; i < vect1.size(); i++) {
		if (!vect1[i].equals(vect2[i]))
			return false;
	}
	return true;
}

// Added by Frederic Desmons (2007/11/14)
int CF_datetimeCmp(int& sign, const string& datetime1, const string& datetime2, string& errorMessage) {
	unsigned long long binaryDatetime1, binaryDatetime2;
	CF_assert(CF_datetimeToBinary(binaryDatetime1, datetime1, errorMessage));
	CF_assert(CF_datetimeToBinary(binaryDatetime2, datetime2, errorMessage));
	if (binaryDatetime1 > binaryDatetime2)
		sign = 1;
	if (binaryDatetime1 < binaryDatetime2)
		sign = -1;
	if (binaryDatetime1 == binaryDatetime2)
		sign = 0;
	return OK;
}


/* ---------------------------------------------------------------------------*/
/*                           CMPI type comparison                             */
/* ---------------------------------------------------------------------------*/

// Added by Frederic Desmons (2007/07/31)
bool CF_CMPIDataCmp(const CMPIBroker* broker, const CMPIData& data1, const CMPIData& data2) {
	if (data1.type != data2.type || data1.state != data2.state)
		return false;
	if ((data1.state & CMPI_goodValue) != CMPI_goodValue && (data1.state & CMPI_keyValue) != CMPI_keyValue)
		return false;
	switch (data1.type) {
		case CMPI_boolean:
			return data1.value.boolean == data2.value.boolean;
		case CMPI_char16:
			return data1.value.char16 == data2.value.char16;
		case CMPI_real32:
			return data1.value.real32 == data2.value.real32;
		case CMPI_real64:
			return data1.value.real64 == data2.value.real64;
		case CMPI_uint8:
			return data1.value.uint8 == data2.value.uint8;
		case CMPI_uint16:
			return data1.value.uint16 == data2.value.uint16;
		case CMPI_uint32:
			return data1.value.uint32 == data2.value.uint32;
		case CMPI_uint64:
			return data1.value.uint64 == data2.value.uint64;
		case CMPI_sint8:
			return data1.value.sint8 == data2.value.sint8;
		case CMPI_sint16:
			return data1.value.sint16 == data2.value.sint16;
		case CMPI_sint32:
			return data1.value.sint32 == data2.value.sint32;
		case CMPI_sint64:
			return data1.value.sint64 == data2.value.sint64;
		case CMPI_ref: {
			Objectpath ob1, ob2;
			if (CT_ToC(broker, data1, ob1) != OK || CT_ToC(broker, data2, ob2) != OK)
				return false;
			return ob1.equals(ob2);
		}
		case CMPI_string: {
			string str1, str2;
			if (CT_ToC(data1, str1) != OK || CT_ToC(data2, str2) != OK)
				return false;
			return CF_strCmpNoCase(str1, str2);
		}
		case CMPI_dateTime: {
			string dt1, dt2;
			if (CT_ToCDatetime(data1, dt1) != OK || CT_ToCDatetime(data2, dt2) != OK)
				return false;
			// modified by Frederic Desmons (2007/11/15)
			int sign;
			string errorMessage;
			int errorCode = CF_datetimeCmp(sign, dt1, dt2, errorMessage);
			if (errorCode != OK)
				return false;
			return sign==0;
			//return CF_strCmpNoCase(dt1, dt2);
		}
	}
	return false;
}


/* ---------------------------------------------------------------------------*/
/*                       Time and Datetime manipulation                       */
/* ---------------------------------------------------------------------------*/

// Added by Frederic Desmons (2007/07/10)
short CF_getCurrentTimeZone() {
	struct timeval tv;
	struct timezone tz;
	signed short os_timezone;
	if(gettimeofday(&tv, &tz) == 0)
		os_timezone = tz.tz_minuteswest*-1;
	else
		os_timezone = 0;
	return os_timezone;
}

// Added by Frederic Desmons (2007/07/10)
time_t CF_getUTCTime() {
	return time(NULL);
}

// Added by Frederic Desmons (2007/07/11)
string CF_toLocalTime(time_t UTC_time) {
	string result;
	struct tm cttm;
	if (localtime_r(&UTC_time, &cttm) != NULL) {
		char* temp = (char*) malloc(22);
		strftime(temp, 22, "%Y%m%d%H%M%S.000000", &cttm);
		result = temp;
		if (temp) free(temp);
	}
	return result;
}

// Added by Frederic Desmons (2007/07/11)
string CF_toLocalTime(time_t UTC_time, short timezone) {
	string result;
	struct tm cttm;
	if (localtime_r(&UTC_time, &cttm) != NULL) {
		char* temp = (char*) malloc(22);
		strftime(temp, 22, "%Y%m%d%H%M%S.000000", &cttm);
		result = temp;
		if (temp) free(temp);
	}
	CF_addTimeZone(result, timezone);
	return result;
}

// Added by Frederic Desmons (2007/07/11)
string CF_getLocalDateTime(time_t UTC_time) {
	return CF_toLocalTime(CF_getUTCTime(), CF_getCurrentTimeZone());
}

// Added by Frederic Desmons (2007/07/10)
void CF_addTimeZone(string& datetime, short timezone) {
	if (timezone>=0)
		datetime += "+";
	else
		datetime += "-";
	datetime += CF_intToStr(abs(timezone));
}

// Added by Ilsoo Byun (2007/06/11)
int CF_lastModified(const string& filepath, string& modifiedTime, string& errorMessage) {
	struct stat attr;
	stat(filepath.c_str(), &attr);
	struct tm* clock = gmtime(&(attr.st_mtime));

	char* result = (char*) malloc(26);
	strftime(result, 26, "%Y%m%d%H%M%S.000000", clock);

	CF_catTimezone(result, CF_getOsTimezone());

	modifiedTime = (string) result;
	free(result);
	return OK;
}

// Added by Frederic Desmons (2007/11/14)
bool CF_isDatetime(const string& myDatetime) {
	if (myDatetime.size() != 25 || (myDatetime[21] != '+' && myDatetime[21] != '-') || myDatetime[14] != '.' || !CF_isNumber(myDatetime.substr(0, 14)) || !CF_isNumber(myDatetime.substr(15, 6)) || !CF_isNumber(myDatetime.substr(22, 3)))
		return false;
	// 01 < month < 12
	if (atoi(myDatetime.substr(4, 2).c_str()) < 1 || atoi(myDatetime.substr(4, 2).c_str()) > 12)
		return false;
	// 01 < day
	if (atoi(myDatetime.substr(6, 2).c_str()) < 1)
		return false;
	// day < 28,29,30,31
	switch (atoi(myDatetime.substr(4, 2).c_str())) {
	case 1:
		if (atoi(myDatetime.substr(6, 2).c_str()) > 31)
			return false;
	case 2:
		if (atoi(myDatetime.substr(0, 4).c_str())%4 == 0) {
			if (atoi(myDatetime.substr(6, 2).c_str()) > 29)
				return false;
		} else {
			if (atoi(myDatetime.substr(6, 2).c_str()) > 28)
				return false;
		}
	case 3:
		if (atoi(myDatetime.substr(6, 2).c_str()) > 31)
			return false;
	case 4:
		if (atoi(myDatetime.substr(6, 2).c_str()) > 30)
			return false;
	case 5:
		if (atoi(myDatetime.substr(6, 2).c_str()) > 31)
			return false;
	case 6:
		if (atoi(myDatetime.substr(6, 2).c_str()) > 30)
			return false;
	case 7:
		if (atoi(myDatetime.substr(6, 2).c_str()) > 31)
			return false;
	case 8:
		if (atoi(myDatetime.substr(6, 2).c_str()) > 31)
			return false;
	case 9:
		if (atoi(myDatetime.substr(6, 2).c_str()) > 30)
			return false;
	case 10:
		if (atoi(myDatetime.substr(6, 2).c_str()) > 31)
			return false;
	case 11:
		if (atoi(myDatetime.substr(6, 2).c_str()) > 30)
			return false;
	case 12:
		if (atoi(myDatetime.substr(6, 2).c_str()) > 31)
			return false;
	}
	// 00 < hour < 23
	if (atoi(myDatetime.substr(8, 2).c_str()) > 23)
		return false;
	// 00 < minutes < 59
	if (atoi(myDatetime.substr(10, 2).c_str()) > 59)
		return false;
	// 00 < seconds < 59
	if (atoi(myDatetime.substr(12, 2).c_str()) > 59)
		return false;

	return true;
}

// Added by Frederic Desmons (2007/11/14)
int CF_datetimeToBinary(unsigned long long& binaryDatetime, const string& myDatetime, string& errorMessage) {
	// check if the datetime is valid
	if (!CF_isDatetime(myDatetime)) {
		errorMessage = "Wrong format: datetime (" + myDatetime + ")";
		binaryDatetime=0;
		return FAILED;
	}

	// convert the datetime to struct tm
	struct tm _time;

	if (atoi(myDatetime.substr(0, 4).c_str()) < 1970) {
		errorMessage = "Only dates after 1970/01/01 are supported";
		binaryDatetime=0;
		return FAILED;
	}

	_time.tm_year=atoi(myDatetime.substr(0, 4).c_str()) - 1900;
	_time.tm_mon=atoi(myDatetime.substr(4, 2).c_str()) - 1;
	_time.tm_mday=atoi(myDatetime.substr(6, 2).c_str());
	_time.tm_hour=atoi(myDatetime.substr(8, 2).c_str());
	_time.tm_min=atoi(myDatetime.substr(10, 2).c_str());
	_time.tm_sec=atoi(myDatetime.substr(12, 2).c_str());
	_time.tm_wday=-1;
	_time.tm_yday=-1;
	_time.tm_isdst=-1;

	time_t seconds = mktime(&_time);

	seconds += CF_getCurrentTimeZone()*60;

	// absolute time from GMT
	if (myDatetime[21] == '+')
		seconds -= atoll(myDatetime.substr(22, 3).c_str())*60;
	else
		seconds += atoll(myDatetime.substr(22, 3).c_str())*60;

	// in microseconds
	binaryDatetime = ((unsigned long long) seconds)*1000*1000 + atoi(myDatetime.substr(15, 6).c_str());

	return OK;
}

// Added by Guillaume Bottex (2008/08/07)
int CF_monthAbvStrToInt(const string& month)
{
  return CF_monthAbvStrToInt(month.c_str());
}

// Added by Guillaume Bottex (2008/08/07)
int CF_monthAbvStrToInt(const char* month)
{
  const char monthlist[][4]={   "JAN","FEB","MAR","APR","MAY","JUN",
                                "JUL","AUG","SEP","OCT","NOV","DEC"};

  char month_upper[4];

  for(int i=0;i<4;i++)
    month_upper[i]=(char)toupper((int)month[i]);

  for(int i=0;i<12;i++)
  {
    if(strcmp(month_upper,monthlist[i])==0)
      return i+1;
  }

  return 0;
}

// Deprecated by Frederic Desmons (2007/07/10)
signed short CF_getOsTimezone() {
	struct timeval  tv;
	struct timezone tz;
	signed short os_timezone;
	if(gettimeofday(&tv, &tz) == 0)
		os_timezone = tz.tz_minuteswest*-1;
	else
		os_timezone = 0;
	return os_timezone;
}

// Deprecated by Frederic Desmons (2007/07/10)
time_t CF_localTime() {
	struct timeval  tv;
	struct timezone tz;
	time_t local_time=0;
	if(gettimeofday(&tv, &tz)==0)
		local_time = tv.tv_sec;
	return local_time;
}

// Deprecated by Frederic Desmons (2007/07/10)
void CF_catTimezone(char *str, signed short zone) {
	char* tz = NULL;
	if((tz=(char*) malloc(sizeof(long long))) != NULL) {
		sprintf(tz, "%+04d", zone);
		if(str != NULL) strcat(str,tz);
		if(tz) free(tz);
	}
}

// Deprecated by Frederic Desmons (2007/07/10)
string CF_timeToString(time_t time) {
	string result;
	char* tm = NULL;
	struct tm cttm;
	if(gmtime_r(&time , &cttm) != NULL) {
		tm = (char*) malloc(26);
		strftime(tm, 26, "%Y%m%d%H%M%S.000000", &cttm);
		CF_catTimezone(tm, CF_getOsTimezone());
		result = tm;
		if (tm) free(tm);
	}
	return result;
}


/* ---------------------------------------------------------------------------*/
/*                         Integer / String conversion                        */
/* ---------------------------------------------------------------------------*/

string CF_intToStr(unsigned char value) {
	ostringstream ostr;
	ostr << (unsigned short) value;
	return ostr.str();
}

string CF_intToStr(unsigned short value) {
	_TO_STR
}

string CF_intToStr(unsigned long value) {
	_TO_STR
}

string CF_intToStr(unsigned long long value) {
	_TO_STR
}

string CF_intToStr(signed char value) {
	ostringstream ostr;
	ostr << (signed short) value;
	return ostr.str();
}

string CF_intToStr(short value) {
	_TO_STR
}

string CF_intToStr(long value) {
	_TO_STR
}

string CF_intToStr(long long value) {
	_TO_STR
}

string CF_intToStr(float value) {
	_TO_STR
}

string CF_intToStr(double value) {
	_TO_STR
}

string CF_intToStr(int value) {
	_TO_STR
}

string CF_intToStr(unsigned int value) {
	_TO_STR
}

string CF_boolToStr(bool value) {
	if (value)
		return "true";
	else
		return "false";
}

unsigned long CF_strToUL(const string& input) {
	const char* begin = input.c_str();
	char* end = (char*) begin[input.size()-1];
	return strtoul(begin, &end, 10);
}

unsigned long long CF_strToULL(const string& input) {
	const char* begin = input.c_str();
	char* end = (char*) begin[input.size()-1];
	return strtoull(begin, &end, 10);
}

// Added by Ilsoo Byun (2007/05/10)
bool CF_isNumber(const string& str) {
	string::size_type i=0;
	for (; i < str.size(); i++) {
		if (!isdigit(str[i]))
			return false;
	}
	return true;
}


/* ---------------------------------------------------------------------------*/
/*                                    Sorting                                 */
/* ---------------------------------------------------------------------------*/

bool CF_foundInSortedList(const string& element, const vector<string>& list, vector<string>::size_type& index) {
	if (list.size()==0) {
	  index=0;
	  return false;
	}
	if (list.size()==1) {
		if (strcmp(list[0].c_str(), element.c_str())==0)
			return true;
		if (strcmp(list[0].c_str(), element.c_str())>0) {
			index=0;
			return false;
		}
		else {
			index=1;
			return false;
		}
	}
	unsigned int begin=0;
	unsigned int end=list.size()-1;
	unsigned int position=(end+begin)/2;
	while(end-begin>1) {
		if (strcmp(list[position].c_str(), element.c_str())>=0)
			end=position;
		else
			begin=position;
		position=(end+begin)/2;
	}
	index=begin;
	if (element==list[begin])
		return true;
	if (element==list[end]) {
		index=end;
		return true;
	}
	if (strcmp(list[end].c_str(), element.c_str())<0)
		index=end+1;
	if (strcmp(list[begin].c_str(), element.c_str())>0)
		index=begin;
	if (strcmp(list[begin].c_str(), element.c_str())<0 && strcmp(list[end].c_str(), element.c_str())>0)
		index=end;
	return false;
}

bool CF_foundInList(const string& element, const vector<string>& list, vector<string>::size_type& index) {
	for (vector<string>::size_type i=0; i<list.size(); i++) {
		if (strcmp(list[i].c_str(), element.c_str())==0) {
			index=i;
			return true;
		}
	}
	return false;
}
