/*
 * This file is part of the KDE Milou Project
 * Copyright (C) 2013  Vishesh Handa <me@vhanda.in>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) version 3, or any
 * later version accepted by the membership of KDE e.V. (or its
 * successor approved by the membership of KDE e.V.), which shall
 * act as a proxy defined in Section 6 of version 3 of the license.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef MILOU_ABSTRACTSOURCE_H
#define MILOU_ABSTRACTSOURCE_H

#include <QObject>
#include <QUrl>

#include "match.h"
#include "context.h"

namespace Milou {

/**
 * \class AbstractSource
 *
 * \brief An AbstractSource is the base class for implementing sources
 * of data. They are similar to runner.
 *
 * A source must implement the query and run functions which are responsible
 * for returns results for a query and for acting on the results.
 *
 * A source should also additionally register the types that it will be
 * returning matches for via the setTypes function
 *
 * \author Vishesh Handa <me@vhanda.in>
 */
class AbstractSource : public QObject
{
    Q_OBJECT
public:
    explicit AbstractSource(QObject* parent = 0);
    virtual ~AbstractSource();

    /**
     * All sources must implement this query function where they process
     * the query and produce the results. The results are returned
     * by adding each of them via the addMatch function.
     *
     * A source may be synchronous or asynchronous. If your source is
     * non-trivial you may want to spawn a thread and avoid blocking the main
     * application
     *
     * \sa addMatch
     */
    virtual void query(const Context& context) = 0;

    /**
     * Perform the default action on a particular match.
     * By default this function does nothing.
     */
    virtual void run(const Match& match);

    void setQueryLimit(int limit);
    int queryLimit();

    /**
     * Returns a list of types that the source can return. These
     * sources should checked before each query.
     */
    QList<MatchType*> types();
    void setTypes(const QList<MatchType*> types);

    /**
     * Stop the last running query. By default this function
     * does not do anything
     */
    virtual void stop();

protected:
    /**
     * Should be called by the sources when a match has been found.
     */
    void addMatch(const Match& match);

signals:
    void matchAdded(const Match& match);

private:
    int m_queryLimit;
    QList<MatchType*> m_types;
};

}

#endif // MILOU_ABSTRACTSOURCE_H
