<?xml version="1.0"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:func="http://exslt.org/functions"
                xmlns:pml="http://wohlberg.net/xml/photoml"
                exclude-result-prefixes="func pml"
                extension-element-prefixes="func"
                version="1.0">

<!-- 
     This file contains templates for generating PhotoML digital
     or digimage elements from the intermediate image data output by
     the pmldigital tool.

     Copyright © 2004-2010 Brendt Wohlberg <photoml@wohlberg.net>

     This is free software; you can redistribute it and/or modify it 
     under the terms of version 2 of the GNU General Public License 
     at http://www.gnu.org/licenses/gpl-2.0.txt.

     This software is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
     GNU General Public License for more details.
-->

  <xsl:output method="xml" indent="yes" encoding="utf-8"
              doctype-public="-//BW//DTD PhotoML 0.13//EN"
              doctype-system="photo.dtd"/>


  <xsl:key name="imagegid" match="image" use="./@group-id"/>

  <xsl:key name="owner" match="MakerNotes.OwnerName" use="."/>

  <xsl:variable name="single-owner" 
                select="//MakerNotes.OwnerName and
                        count(key('owner',//image[1]/MakerNotes.OwnerName)) = 
                        count(//image)"/>
  
  <xsl:key name="camera" match="image" 
           use="concat(EXIF.Make,EXIF.Model)"/>

  <xsl:variable name="single-camera" 
                select="//image[child::EXIF.Make and child::EXIF.Model] and
                        count(key('camera',concat(//image[1]/EXIF.Make,
                                                  //image[1]/EXIF.Model[1]))) =
                        count(//image)"/> 

  <xsl:key name="lens" match="Composite.LensID" use="."/>

  <xsl:variable name="single-lens" 
                select="//image[child::Composite.LensID] and
                        count(key('lens',//image[1]/Composite.LensID)) =
                        count(//image)"/> 


  <!-- Match top level element of pmldigital image data representation. -->
  <xsl:template match="/pmldigital">
    <photo>
      <xsl:if test="//image[not(@scan='yes')]">
        <xsl:choose>
          <xsl:when test="@group-id or not(image[@group-id])">
            <digital id="to-be-assigned">
              <xsl:if test="@group-id">
                <xsl:attribute name="id">
                  <xsl:value-of select="@group-id"/>
                </xsl:attribute>
              </xsl:if>
              <xsl:call-template name="common-defaults"/>
              <xsl:apply-templates select="image[not(@scan='yes')]"/>
            </digital>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="common-defaults"/>
            <xsl:for-each select="image[not(@scan='yes')]
                [generate-id(.)=generate-id(key('imagegid',./@group-id)[1])]">
              <xsl:sort data-type="text" order="ascending" select="@group-id"/>
              <xsl:variable name="gid" select="@group-id"/>
              <digital id="{$gid}">
                <xsl:for-each select="//image[not(@scan='yes')]
                                             [@group-id=$gid]">
                  <xsl:sort data-type="number" order="ascending" 
                            select="@frame-id"/>
                  <xsl:apply-templates select="."/>
                </xsl:for-each>
              </digital>
            </xsl:for-each>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:if>
      <xsl:if test="//image[@scan='yes']">
        <!-- Create a blank line -->
        <xsl:comment/>
        <xsl:apply-templates select="image[@scan='yes']"/>
      </xsl:if>
    </photo>
  </xsl:template>


  <!-- Insert defaults for properties common to all frames. -->
  <xsl:template name="common-defaults">
    <xsl:if test="$single-owner or $single-camera">
      <!-- Create a blank line -->
      <xsl:comment/>
      <defaults>
        <frame>
          <xsl:if test="$single-owner">
            <photographer>
              <name>
                <xsl:value-of 
                  select="//MakerNotes.OwnerName"/>
              </name>
            </photographer>
          </xsl:if>
          <xsl:if test="$single-camera or $single-lens">
            <equipment>
              <xsl:if test="$single-camera">
                <body>
                  <xsl:apply-templates select="/pmldigital/image[1]
                                               /EXIF.Make"/>
                  <xsl:apply-templates select="/pmldigital/image[1]
                                               /EXIF.Model"/>
                </body>
              </xsl:if>
              <xsl:if test="$single-lens">
                <lens>
                  <xsl:apply-templates select="/pmldigital/image[1]
                                               /Composite.LensID"/>
                </lens>
              </xsl:if>
            </equipment>
          </xsl:if>
        </frame>
      </defaults>
    </xsl:if>
    <!-- Create a blank line -->
    <xsl:comment/>
  </xsl:template>


  <!-- Deal with EXIF image data from images from a digital camera. -->
  <xsl:template match="image[not(@scan='yes')]">
    <xsl:variable name="fid">
      <xsl:choose>
        <xsl:when test="@frame-id">
          <xsl:value-of select="@frame-id"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="format-number(position(),'000')"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <frame id="{$fid}">
      <xsl:if test="not($single-owner) and MakerNotes.OwnerName">
        <photographer>
          <name>
            <xsl:value-of select="MakerNotes.OwnerName"/>
          </name>
        </photographer>
      </xsl:if>
      <xsl:if test="EXIF.DateTimeOriginal or XMP.DateTimeOriginal or 
                    EXIF.GPSLatitude or XMP.GPSLatitude or XMP.Country">
        <scene>
          <xsl:if test="EXIF.DateTimeOriginal or XMP.DateTimeOriginal">
            <occasion>
              <xsl:choose>
                <xsl:when test="XMP.DateTimeOriginal">
                   <xsl:apply-templates select="XMP.DateTimeOriginal"/>
                </xsl:when>
                <xsl:otherwise>
                   <xsl:apply-templates select="EXIF.DateTimeOriginal"/>
                </xsl:otherwise>
              </xsl:choose>
            </occasion>
          </xsl:if>
          <xsl:if test="EXIF.GPSLatitude or XMP.GPSLatitude or XMP.Country">
            <location>
              <xsl:apply-templates select="XMP.Country"/>
              <xsl:apply-templates select="XMP.State"/>
              <xsl:apply-templates select="XMP.City"/>
              <xsl:apply-templates select="XMP.Location"/>
              <xsl:if test="EXIF.GPSLatitude or XMP.GPSLatitude">
                <coordinates>
                  <xsl:if test="EXIF.GPSMapDatum or XMP.GPSMapDatum">
                    <xsl:attribute name="datum">
                      <xsl:value-of select="*[substring-after(
                        name(.),'.')='GPSMapDatum'][1]"/>
                    </xsl:attribute>
                  </xsl:if>
                  <latitude>
                    <xsl:if test="*[substring-after(name(.),'.')=
                                  'GPSLatitudeRef'] = 'South'">
                      <xsl:text>-</xsl:text>
                    </xsl:if>
                    <xsl:value-of select="pml:firstword(
                        *[substring-after(name(.),'.')='GPSLatitude'][1])"/>
                  </latitude>
                  <longitude>
                    <xsl:if test="*[substring-after(name(.),'.')=
                                  'GPSLongitudeRef'] = 'West'">
                      <xsl:text>-</xsl:text>
                    </xsl:if>
                    <xsl:value-of select="pml:firstword(
                          *[substring-after(name(.),'.')='GPSLongitude'][1])"/>
                  </longitude>
                  <xsl:if test="*[substring-after(name(.),'.')='GPSAltitude']">
                    <altitude reference='amsl'>
                      <xsl:if test="*[substring-after(name(.),'.')=
                                    'GPSAltitude'] = 'Below Sea Level'">
                        <xsl:text>-</xsl:text>
                      </xsl:if>
                      <xsl:value-of
                       select="substring-before(*[substring-after(name(.),'.')=
                                                'GPSAltitude'][1], ' ')"/>
                  </altitude>
                </xsl:if>
              </coordinates>
              </xsl:if>
            </location>
          </xsl:if>
        </scene>
      </xsl:if>
      <xsl:if test="(not($single-camera) and EXIF.Make|EXIF.Model) or
                    (not($single-lens) and Composite.LensID)">
        <equipment>
          <xsl:if test="not($single-camera) and EXIF.Make|EXIF.Model">
            <body>
              <xsl:apply-templates select="EXIF.Make"/>
              <xsl:apply-templates select="EXIF.Model"/>
            </body>
          </xsl:if>
          <xsl:if test="not($single-lens) and Composite.LensID">
            <lens>
              <xsl:apply-templates select="Composite.LensID"/>
            </lens>
          </xsl:if>
        </equipment>
      </xsl:if>
      <xsl:if test="EXIF.FNumber|MakerNotes.FNumber|
                    MakerNotes.ExposureTime|EXIF.ExposureTime|
                    EXIF.ExposureMode|
                    EXIF.ExposureCompensation|EXIF.Flash|
                    MakerNotes.FlashExposureComp|
                    MakerNotes.ISO|Composite.ISO|
                    EXIF.MeteringMode">
        <exposure>
          <xsl:apply-templates select="pml:preference(.,
                                      'MakerNotes.FNumber,EXIF.FNumber')"/>
          <xsl:apply-templates select="pml:preference(.,
                                'MakerNotes.ExposureTime,EXIF.ExposureTime')"/>
          <xsl:apply-templates select="EXIF.ExposureCompensation"/>
          <xsl:apply-templates select="EXIF.Flash"/>
          <xsl:apply-templates select="MakerNotes.FlashExposureComp[1]"/>
          <xsl:apply-templates select="pml:preference(.,
                                       'MakerNotes.ISO,Composite.ISO')"/>
          <xsl:apply-templates select="EXIF.ExposureMode"/>
          <xsl:apply-templates select="EXIF.MeteringMode"/>
        </exposure>
      </xsl:if>
      <xsl:if test="EXIF.FocalLength|MakerNotes.FocusDistance">
        <focus>
          <xsl:apply-templates select="pml:preference(.,
                                 'MakerNotes.FocalLength,EXIF.FocalLength')"/>
          <xsl:apply-templates select="MakerNotes.FocusDistance"/>
        </focus>
      </xsl:if>
      <digimage href="{@name}">
        <properties>
          <width>
            <xsl:value-of select="pml:preference(.,
                 'File.ImageWidth,EXIF.ImageWidth,MakerNotes.ImageWidth')"/>
          </width>
          <height>
            <xsl:value-of select="pml:preference(.,
                 'File.ImageHeight,EXIF.ImageHeight,MakerNotes.ImageHeight')"/>
          </height>
          <xsl:apply-templates select="File.FileType"/>
          <xsl:if test="EXIF.ColorSpace">
            <icc-profile>
              <xsl:value-of select="EXIF.ColorSpace"/>
            </icc-profile>
          </xsl:if>
          <xsl:if test="@file-hash">
            <file-hash>
              <xsl:value-of select="@file-hash"/>
            </file-hash>
          </xsl:if>
          <xsl:if test="@image-hash">
            <image-hash>
              <xsl:value-of select="@image-hash"/>
            </image-hash>
          </xsl:if>
        </properties>
      </digimage>
    </frame>
    <!-- Create a blank line -->
    <xsl:comment/>
  </xsl:template>


  <!-- Handle EXIF image creation information. -->
  <xsl:template match="EXIF.DateTimeOriginal|EXIF.CreateDate|EXIF.ModifyDate|
                       XMP.DateTimeOriginal|XMP.ModifyDate">
    <xsl:variable name="tstr" 
                  select="translate(substring-after(.,' '),'.',':')"/>
    <xsl:variable name="zone" 
                  select="substring-after(translate($tstr,'-','+'),'+')"/>
    <date>
      <xsl:value-of select="translate(substring-before(.,' '),':.','--')"/>
    </date>
    <time>
      <xsl:if test="$zone != ''">
        <xsl:attribute name="zone">
          <xsl:choose>
            <xsl:when test="contains($tstr,'+')">
              <xsl:text>+</xsl:text>
            </xsl:when>
            <xsl:otherwise>
              <xsl:text>-</xsl:text>
            </xsl:otherwise>
          </xsl:choose>
          <xsl:value-of select="$zone"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:choose>
        <xsl:when test="$zone = ''">
          <xsl:value-of select="$tstr"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="substring-before(translate($tstr,'-','+'),
                                '+')"/>
        </xsl:otherwise>
      </xsl:choose>
    </time>
  </xsl:template>


  <!-- Handle XMP country information -->
  <xsl:template match="XMP.Country">
    <country>
      <xsl:value-of select="."/>
    </country>
  </xsl:template>


  <!-- Handle XMP state information -->
  <xsl:template match="XMP.State">
    <region>
      <xsl:value-of select="."/>
    </region>
  </xsl:template>


  <!-- Handle XMP city information -->
  <xsl:template match="XMP.City">
    <city>
      <xsl:value-of select="."/>
    </city>
  </xsl:template>


  <!-- Handle XMP location information -->
  <xsl:template match="XMP.Location">
    <place>
      <xsl:value-of select="."/>
    </place>
  </xsl:template>


  <!-- Handle EXIF exposure time information. -->
  <xsl:template match="MakerNotes.ExposureTime|EXIF.ExposureTime">
    <shutter>
      <xsl:value-of select="."/>
    </shutter>
  </xsl:template>


  <!-- Handle EXIF aperture information. -->
  <xsl:template match="MakerNotes.FNumber|EXIF.FNumber">
    <aperture>
      <xsl:value-of select="."/>
    </aperture>
  </xsl:template>


  <!-- Handle EXIF focal length information. -->
  <xsl:template match="MakerNotes.FocalLength|EXIF.FocalLength">
    <!--<xsl:variable name="units" select="substring-after(.,' ')"/>-->
    <xsl:variable name="units" select="translate(.,'0123456789. ','')"/>
    <focal-length>
      <xsl:if test="$units != 'mm'">
        <xsl:attribute name="units">
          <xsl:value-of select="$units"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:value-of select="translate(substring-before(.,$units),' ','')"/>
    </focal-length>
  </xsl:template>


  <!-- Handle EXIF subject distance information. -->
  <xsl:template match="MakerNotes.FocusDistance">
    <xsl:variable name="units" select="translate(.,'0123456789. ','')"/>
    <focal-distance>
      <xsl:if test="$units != 'm'">
        <xsl:attribute name="units">
          <xsl:value-of select="$units"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:value-of select="translate(substring-before(.,$units),' ','')"/>
    </focal-distance>
  </xsl:template>


  <!-- Handle EXIF exposure compensation information. -->
  <xsl:template match="EXIF.ExposureCompensation">
    <exp-comp>
      <xsl:value-of select="."/>
    </exp-comp>
  </xsl:template>


  <!-- Handle EXIF flash usage information. -->
  <xsl:template match="EXIF.Flash">
    <flash>
      <xsl:value-of select="."/>
    </flash>
  </xsl:template>


  <!-- Handle EXIF flash exposure compensation information. -->
  <xsl:template match="MakerNotes.FlashExposureComp">
    <flash-comp>
      <xsl:value-of select="."/>
    </flash-comp>
  </xsl:template>


  <!-- Handle EXIF rated ISO speed information. -->
  <xsl:template match="MakerNotes.ISO|Composite.ISO">
    <!-- Construct a rated-speed element if a numeric value is
         available (as opposed to "Auto" etc. -->
    <xsl:if test="translate(.,'0123456789','')=''">
      <rated-speed>
        <xsl:value-of select="."/>
      </rated-speed>
    </xsl:if>
  </xsl:template>


  <!-- Handle EXIF exposure mode information. -->
  <xsl:template match="EXIF.ExposureMode">
    <mode>
      <xsl:value-of select="."/>
    </mode>
  </xsl:template>
  

  <!-- Handle EXIF metering mode information. -->
  <xsl:template match="EXIF.MeteringMode">
    <metering>
      <xsl:value-of select="."/>
    </metering>
  </xsl:template>


  <!-- Handle EXIF bit depth information. -->
  <xsl:template match="EXIF.BitsPerSample">
    <bit-depth>
      <xsl:value-of select="pml:firstword(.)"/>
    </bit-depth>
  </xsl:template>


  <!-- Handle EXIF camera manufacturer information. -->
  <xsl:template match="EXIF.Make">
    <xsl:variable name="sigpart">
      <xsl:choose>
        <xsl:when test="contains(.,'CORPORATION')">
          <xsl:value-of select="normalize-space(
                                  substring-before(.,'CORPORATION'))"/>
        </xsl:when>
        <xsl:when test="contains(.,'Camera')">
          <xsl:value-of select="normalize-space(
                                  substring-before(.,'Camera'))"/>
        </xsl:when>
        <xsl:when test="contains(.,'OPTICAL')">
          <xsl:value-of select="normalize-space(
                                  substring-before(.,'OPTICAL'))"/>
        </xsl:when>
        <xsl:when test="contains(.,'Electric')">
          <xsl:value-of select="normalize-space(
                                  substring-before(.,'Electric'))"/>
        </xsl:when>
        <xsl:when test="contains(.,'Co.')">
          <xsl:value-of select="normalize-space(
                                  substring-before(.,'Co.'))"/>
        </xsl:when>
        <xsl:when test="contains(.,'CO.')">
          <xsl:value-of select="normalize-space(
                                  substring-before(.,'CO.'))"/>
        </xsl:when>
        <xsl:when test="contains(.,'Company')">
          <xsl:value-of select="normalize-space(
                                  substring-before(.,'Company'))"/>
        </xsl:when>
        <xsl:when test="contains(.,'COMPANY')">
          <xsl:value-of select="normalize-space(
                                  substring-before(.,'COMPANY'))"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="."/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <make>
      <xsl:value-of select="pml:lclast($sigpart)"/>
    </make>
  </xsl:template>


  <!-- Handle EXIF camera model information. -->
  <xsl:template match="EXIF.Model">
    <xsl:variable name="mkstrlen"
                  select="number(contains(../EXIF.Make,' '))*
                      string-length(substring-before(../EXIF.Make,' ')) +
                          number(not(contains(../EXIF.Make,' ')))*
                      string-length(../EXIF.Make)"/>
    <model>
      <xsl:choose>
        <xsl:when test="substring(../EXIF.Make, 1, $mkstrlen) = 
                        substring(., 1, $mkstrlen)">
          <xsl:value-of select="normalize-space(substring(.,$mkstrlen+1))"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="."/>
        </xsl:otherwise>
      </xsl:choose>
    </model>
  </xsl:template>


  <!-- Handle EXIF lens information. -->
  <xsl:template match="Composite.LensID">

    <xsl:variable name="make0" select="pml:contained(.,
                 'Canon|Cosina|Nikkor|Sigma|Soligor|Tamron|Tokina|Unknown')"/>
    <xsl:variable name="make1" select="pml:contained(.,
                 'Canon|Cosina|IX-Nikkor|Micro-Nikkor|Zoom-Nikkor|
                  Nikkor|Sigma|Soligor|Tamron|Tokina|Unknown')"/>

    <xsl:choose>
      <xsl:when test="$make0 != ''">
        <make>
          <xsl:value-of select="$make0"/>
        </make>
        <model>
          <xsl:value-of select="concat(
                                normalize-space(substring-before(.,$make1)),
                                ' ',
                                normalize-space(substring-after(.,$make1))
                                )"/>
        </model>
      </xsl:when>
      <xsl:otherwise>
        <model>
          <xsl:value-of select="."/>
        </model>
      </xsl:otherwise>
    </xsl:choose>
      
  </xsl:template>


  <!-- Deal with image data from digital images from film scanner. -->
  <xsl:template match="image[@scan='yes']">
    <digimage href="{@name}">
      <xsl:if test="@group-id">
        <xsl:attribute name="group-id">
          <xsl:value-of select="@group-id"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@frame-id">
        <xsl:attribute name="frame-id">
          <xsl:value-of select="@frame-id"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@image-id">
        <xsl:attribute name="image-id">
          <xsl:value-of select="@image-id"/>
        </xsl:attribute>
      </xsl:if>
      <properties>
        <width>
          <xsl:value-of select="pml:preference(.,
               'File.ImageWidth,EXIF.ImageWidth,MakerNotes.ImageWidth')"/>
        </width>
        <height>
          <xsl:value-of select="pml:preference(.,
               'File.ImageHeight,EXIF.ImageHeight,MakerNotes.ImageHeight')"/>
        </height>
        <xsl:apply-templates select="EXIF.BitsPerSample[1]"/>
        <xsl:apply-templates select="File.FileType"/>
        <xsl:if test="@file-hash">
          <file-hash>
            <xsl:value-of select="@file-hash"/>
          </file-hash>
        </xsl:if>
        <xsl:if test="@image-hash">
          <image-hash>
            <xsl:value-of select="@image-hash"/>
          </image-hash>
        </xsl:if>
      </properties>
      <history>
        <origin>
          <xsl:if test="EXIF.ModifyDate">
            <xsl:apply-templates select="EXIF.ModifyDate"/>
          </xsl:if>
          <scanner>
            <xsl:if test="EXIF.Make">
              <hardware>
                <make>
                  <xsl:value-of select="EXIF.Make"/>
                </make>
                <xsl:apply-templates select="EXIF.Model"/>
              </hardware>
            </xsl:if>
            <xsl:if test="EXIF.Software">
              <software>
                <publisher>
                  <xsl:value-of select="substring-before(EXIF.Software,' ')"/>
                </publisher>
                <name>
                  <xsl:value-of select="substring-before(
                                          substring-after(EXIF.Software,' '), 
                                          ' ')"/>
                </name>
                <version>
                  <xsl:value-of select="substring-after(
                                          substring-after(EXIF.Software,' '),
                                          ' ')"/>
                </version>
              </software>
            </xsl:if>
            <xsl:if test="MakerNotes.FilmType|MakerNotes.BitDepth|
                          MakerNotes.MasterGain|MakerNotes.ColorGain|
                          *[substring(name(.),1,18)='MakerNotes.Digital']">
              <settings>
                <xsl:apply-templates select="MakerNotes.FilmType"/>
                <xsl:apply-templates select="MakerNotes.BitDepth"/>
                <xsl:apply-templates select="MakerNotes.MasterGain"/>
                <xsl:apply-templates select="MakerNotes.ColorGain"/>
                <xsl:apply-templates select="*[substring(name(.),1,18)=
                                             'MakerNotes.Digital']"/>
              </settings>
            </xsl:if>
          </scanner>
        </origin>
      </history>
    </digimage>
    <!-- Create a blank line -->
    <xsl:comment/>
  </xsl:template>

  
  <!-- Handle scanner analog gain settings. -->
  <xsl:template match="MakerNotes.MasterGain">
    <setting name="MasterGain">
      <xsl:value-of select="."/>
    </setting>
  </xsl:template>

  <!-- Handle scanner analog gain settings. -->
  <xsl:template match="MakerNotes.ColorGain">
    <setting name="ColorGain">
      <xsl:value-of select="."/>
    </setting>
  </xsl:template>


  <!-- Handle scanner digital postprocessing settings. -->
  <xsl:template match="*[substring(name(.),1,18)='MakerNotes.Digital']">
    <setting>
      <xsl:attribute name="name">
        <xsl:value-of select="substring(name(),12,18)"/>
      </xsl:attribute>
      <xsl:value-of select="."/>
    </setting>
  </xsl:template>


  <!-- Handle scanner film type setting. -->
  <xsl:template match="MakerNotes.FilmType">
    <setting name='film type'>
      <xsl:value-of select="."/>
    </setting>
  </xsl:template>


  <!-- Handle scanner bit depth setting. -->
  <xsl:template match="MakerNotes.BitDepth">
    <setting name='bit depth'>
      <xsl:value-of select="."/>
    </setting>
  </xsl:template>


  <!-- Handle image format. -->
  <xsl:template match="File.FileType">
    <file-format>
      <xsl:variable name="lc" select="pml:lc(.)"/>
      <xsl:choose>
        <xsl:when test="$lc = 'bmp' or $lc = 'jpeg' or $lc = 'j2k' or
                        $lc = 'png' or $lc = 'psd'  or $lc = 'tiff'">
          <xsl:attribute name="std">
            <xsl:value-of select="$lc"/>
          </xsl:attribute>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="."/>
        </xsl:otherwise>
      </xsl:choose>
    </file-format>
  </xsl:template>


  <!-- Extension function to lower-case a string. -->
  <func:function name="pml:lc">
    <xsl:param name="x" select="''"/>

    <func:result select="translate($x, 'ABCDEFGHIJKLMNOPQRSTUVWXYZ',
                                       'abcdefghijklmnopqrstuvwxyz')"/>
  </func:function>


  <!-- Extension function to lower-case all but first character in each
       word in a string. -->
  <func:function name="pml:lclast">
    <xsl:param name="x" select="''"/>

    <xsl:choose>
      <xsl:when test="contains($x,' ')">
        <func:result select="concat(pml:lclast(substring-before($x,' ')),' ',
                                     pml:lclast(substring-after($x,' ')))"/>
      </xsl:when>
      <xsl:otherwise>
        <func:result select="concat(substring($x, 1, 1),
                                     pml:lc(substring($x, 2)))"/>
      </xsl:otherwise>
    </xsl:choose>

  </func:function>


  <!-- Extension function returning the first word in a string. -->
  <func:function name="pml:firstword">
    <xsl:param name="x" select="''"/>

    <xsl:choose>
      <xsl:when test="contains($x,' ')">
        <func:result select="substring-before($x,' ')"/>
      </xsl:when>
      <xsl:otherwise>
        <func:result select="$x"/>
      </xsl:otherwise>
    </xsl:choose>

  </func:function>


  <!-- Extension function to return the first matching substring in
       a list of substrings. -->
  <func:function name="pml:contained">
    <xsl:param name="str" select="''"/>
    <xsl:param name="lst" select="''"/>

    <xsl:choose>
      <xsl:when test="contains($lst,'|')">
        <xsl:variable name="lst0" 
                      select="normalize-space(substring-before($lst,'|'))"/>
        <xsl:choose>
          <xsl:when test="contains($str, $lst0)">
            <func:result select="$lst0"/>
          </xsl:when>
          <xsl:otherwise>
            <func:result select="pml:contained($str,
                                               substring-after($lst,'|'))"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <xsl:choose>
          <xsl:when test="contains($str, normalize-space($lst))">
            <func:result select="normalize-space($lst)"/>
          </xsl:when>
          <xsl:otherwise>
            <func:result select="''"/>
          </xsl:otherwise>
        </xsl:choose>  
      </xsl:otherwise>
    </xsl:choose>

  </func:function>


  <!-- Extension function returning the value of the first matched
       element name in the 'slct' comma-separated list -->
  <func:function name="pml:preference">
    <xsl:param name="ndst" select="/.."/>
    <xsl:param name="slct" select="''"/>

    <xsl:choose>
      <xsl:when test="contains($slct,',')">
        <xsl:variable name="slct0"
          select="translate(substring-before($slct,','),' ','')"/>
        <xsl:choose>
          <xsl:when test="$ndst/*[name() = $slct0]">
            <func:result select="$ndst/*[name() = $slct0]"/>
          </xsl:when>
          <xsl:otherwise>
            <func:result select="pml:preference($ndst,
                                 substring-after($slct,','))"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <func:result select="$ndst/*[name() = translate($slct,' ','')]"/>
      </xsl:otherwise>
    </xsl:choose>

  </func:function>



</xsl:stylesheet>
