/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower86@gmail.com                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/
#include "userstatistics.h"

#if defined(Q_OS_LINUX)
#  include <sys/utsname.h>
#  include <QFile>
#  include <QStringList>
#  include <QTextStream>
#endif

#if defined(Q_OS_WIN)
#  include <windows.h>
#  include <winbase.h>
#endif

#include <QSysInfo>
#include <QObject>

UserStatistics::UserStatistics() :
  m_os(QObject::tr("Unknown", "OS")),
  m_uptime(-1),
  m_chatTime(0)
{
}


UserStatistics::~UserStatistics()
{
}

QString UserStatistics::OsString()
{
#if defined(Q_OS_LINUX)
  utsname buf;

  if(!uname(&buf))
    return QString("GNU/") + buf.sysname + " (Linux " + buf.release + ")";
  else
    return "GNU/Linux";
#else
#if defined(Q_OS_MAC)
  switch(QSysInfo::MacintoshVersion)
  {
    case QSysInfo::MV_CHEETAH : return "MacOS 10.0(Cheetah)";
    case QSysInfo::MV_PUMA    : return "MacOS 10.1(Puma)";
    case QSysInfo::MV_JAGUAR  : return "MacOS 10.2(Jaguar)";
    case QSysInfo::MV_PANTHER : return "MacOS 10.3(Panther)";
    case QSysInfo::MV_TIGER   : return "MacOS 10.4(Tiger)";
    case QSysInfo::MV_LEOPARD : return "MacOS 10.5(Leopard)";
    case QSysInfo::MV_Unknown : return "MacOS(unknown)";
    default : return "MacOS(unknown)";
  }
#else
#if defined(Q_OS_WIN)
  switch(QSysInfo::WindowsVersion)
  {
    case QSysInfo::WV_32s   : return "Windows 3.1 with Win 32s";
    case QSysInfo::WV_95    : return "Windows 95";
    case QSysInfo::WV_98    : return "Windows 98";
    case QSysInfo::WV_Me    : return "Windows Me";

    case QSysInfo::WV_NT    : return "Windows NT";
    case QSysInfo::WV_2000  : return "Windows 2000";
    case QSysInfo::WV_XP    : return "Windows XP";
    case QSysInfo::WV_2003  : return "Windows Server 2003";
    case QSysInfo::WV_VISTA : return "Windows Vista";

    case QSysInfo::WV_CE    : return "Windows CE";
    case QSysInfo::WV_CENET : return "Windows CE .NET";
    default : return "Windows(unknown)";
  }
#else
  return "Unknown";
#endif

#endif

#endif
}

qint32 UserStatistics::getUptime()
{
#if defined(Q_OS_LINUX)
  QFile file;
  QString ut;
  qint32 res = -1;

  file.setFileName("/proc/uptime");
  if(!file.open(QIODevice::ReadOnly | QIODevice::Text))
  {
    qWarning("[UserStatistics::getUptime]: couldn't open /proc/uptime");
    return -1;
  }

  QTextStream in(&file);
  ut = in.readLine();

  file.close();

  res = (int)(ut.split(" ")[0]).toDouble();

  return res;
#else
#if defined(Q_OS_MAC)
  return -1;

#else
#if defined(Q_OS_WIN)
  return GetTickCount() / 1000;

#else
  return -1;

#endif

#endif

#endif
}

QString UserStatistics::time2string(quint64 tm)
{
  QString str("");
  qint32 s;
  qint32 m;
  qint32 h;
  qint32 d;

  d = tm / (60 * 60 * 24);
  tm -= d * 60 * 60 * 24;
  h = tm / (60 * 60);
  tm -= h * 60 * 60;
  m = tm / 60;
  tm -= m * 60;
  s = tm;

  if(d)
    str += QString().setNum(d) + QObject::tr("d ", "day");
  str += QString().setNum(h) + QObject::tr("h:", "hour");
  str += QString().setNum(m) + QObject::tr("m:", "minute");
  str += QString().setNum(s) + QObject::tr("s", "second");

  return str;
}
