import numpy
import froutines

_symeig_pfx = {'f':'ssy', 'd':'dsy', 'F':'che', 'D':'zhe'}

class SymeigException(Exception):
    pass

class LeadingMinorException(SymeigException):
    pass

# definition of symmetric eigenvalue solver
def symeig(A,B=None,eigenvectors=1,turbo="on",range=None,type=1,overwrite=False):
    """Solve standard and generalized eigenvalue problem for symmetric
and hermitian matrices.

    Syntax:

      w,Z = symeig(A) 
      w = symeig(A,eigenvectors=0)
      w,Z = symeig(A,range=(lo,hi))
      w,Z = symeig(A,B,range=(lo,hi))

    Inputs:

      A     -- An N x N real symmetric or complex hermitian matrix.
      B     -- An N x N real symmetric or complex hermitian definite
               positive matrix.
      eigenvectors -- if set return eigenvalues and eigenvectors, otherwise
                      only eigenvalues 
      turbo -- (only for generalized eigenvalue problem and if range=None)
               if turbo = "on", use divide and conquer algorithm
               (faster but expensive in memory)
      range -- the tuple (lo,hi) represent the indexes of the smallest and
               largest (in ascending order) eigenvalues to be returned.
               1 <= lo < hi <= N
               if range = None, returns all eigenvalues and eigenvectors. 
      type  -- (only for generalized eigenvalue problem)
               Specifies the problem type to be solved:
                      type = 1:  A*x = (lambda)*B*x
                           = 2:  A*B*x = (lambda)*x
                           = 3:  B*A*x = (lambda)*x
      overwrite -- if 'overwrite' is set, computations are done inplace,
                   A and B are overwritten during calculation (you save
                   memory but loose the matrices).
                   If matrices are complex this argument is ignored.
      
    Outputs:

      w     -- (selected) eigenvalues in ascending order.
      Z     -- if range = None, Z contains the matrix of eigenvectors,
               normalized as follows:
                  Z^H * A * Z = lambda and
                    - type = 1 or 2: Z^H * B * Z = I
                    - type = 3     : Z^H * B^(-1) * Z = I
               where ^H means conjugate transpose.
               if range, an N x M matrix containing the orthonormal
               eigenvectors of the matrix A corresponding to the selected
               eigenvalues, with the i-th column of Z holding the eigenvector
               associated with w[i]. The eigenvectors are normalized as above.
    """
    # check dimensions:
    if (len(A.shape) != 2) or (A.shape[0] != A.shape[1]):
        msg = "wrong A dimensions %s, should be (n, n)"%str(A.shape)
        raise SymeigException, msg
    if (B is not None) and (B.shape != A.shape):
        msg = "wrong B dimensions: "+str(B.shape)
        raise SymeigException, msg
        
    
    # Set job for fortran routines
    _job = (eigenvectors and 'V') or 'N'

    # apply conversion rules to find the common conversion type
    dtype = _greatest_common_dtype([A,B])

    # cast matrices to have the same dtype
    A = _refcast(A,dtype)
    B = _refcast(B,dtype)

    # spare a copy if we can
    #if A.flags.c_contiguous and not numpy.iscomplexobj(A):
    #    A = A.T
    #if B is not None and A.flags.c_contiguous and not numpy.iscomplexobj(B):
    #    B = B.T

    # sanitize range
    range = _sanitize_range(A.shape[0], range)
    
    #  Standard Eigenvalue Problem
    #  Use '*evr' routines
    if B is None:
        evr = get_froutines_func("evr",_symeig_pfx,dtype)
	if range is None:
            w,Z,info = evr(A,jobz=_job,range="A",il=1,iu=A.shape[0],
                           overwrite_a=overwrite)
        else: 
            (lo,hi)=range
            w_tot,Z,info = evr(A,jobz=_job,range="I",il=lo,iu=hi,
                               overwrite_a=overwrite)
            w=w_tot[0:hi-lo+1]

    # Generalized Eigenvalue Problem
    else:
        # Use '*gvx' routines if range is specified
        if range is not None:
            gvx = get_froutines_func("gvx",_symeig_pfx,dtype)
            (lo,hi)=range
            w_tot,Z,ifail,info = gvx(A,B,iu=hi,itype=type,jobz=_job,
                                     il=lo, overwrite_a=overwrite,
                                     overwrite_b = overwrite)
            w=w_tot[0:hi-lo+1]
        # Use '*gvd' routine if turbo is on and no range is specified
        elif turbo == "on":
            gvd = get_froutines_func("gvd",_symeig_pfx,dtype)
            Z, w,info = gvd(A,B,itype=type,jobz=_job,
                            overwrite_a=overwrite,
                            overwrite_b = overwrite)
        # Use '*gv' routine if turbo is off and no range is specified
        else:
            gv = get_froutines_func("gv",_symeig_pfx,dtype)
            Z, w,info = gv(A,B,itype=type,jobz=_job,
                           overwrite_a=overwrite,
                           overwrite_b = overwrite)

    # Check if we had a  successful exit
    # Success
    if info == 0:
        if eigenvectors:
            return w,Z
        else:
            return w
    # Catch failures
    # Internal errors: these should never happen.
    # If you get these you found a bug in symeig!
    elif info < 0:
        exceptstring = "illegal value in %i-th argument of internal"%(-info)+\
                       " fortran routine."
        raise SymeigException,exceptstring     
    elif info > 0 and B==None:
        exceptstring = "unrecoverable internal error."
        raise SymeigException,exceptstring
    # The algorithm failed to converge.
    elif info > 0 and info <= B.shape[0]:
        if range is not None:
            exceptstring="the eigenvectors %s"%(numpy.nonzero(ifail))+\
                          " failed to converge."
            raise SymeigException,exceptstring
        else:
            exceptstring="internal fortran routine failed to converge: "+\
                         "%i off-diagonal elements of an "%(info)+\
                         "intermediate tridiagonal form did not converge"+\
                         " to zero."
        raise SymeigException,exceptstring
    # This occurs when B is not positive definite
    else:
        exceptstring="the leading minor of order %i"%(info-B.shape[0])+\
                      " of B is not positive definite. The"+\
                      " factorization of B could not be completed"+\
                      " and no eigenvalues or eigenvectors were computed."
        raise LeadingMinorException,exceptstring


def _sanitize_range(n, range):
    if range is None: return None
    (lo,hi) = range
    if lo < 1:
        lo = 1
    if lo > n:
        lo = n
    if hi > n:
        hi = n
    if lo > hi:
        lo, hi = hi, lo
    return (lo, hi)

def _refcast(array, dtype):
    """
    Cast the array to dtype only if necessary, otherwise return a reference.
    """
    if array is None: return None
    dtype = numpy.dtype(dtype)
    if array.dtype == dtype:
        return array
    return array.astype(dtype)


_type_conv = {('f','d'): 'd', ('f','F'): 'F', ('f','D'): 'D',
              ('d','F'): 'D', ('d','D'): 'D',
              ('F','d'): 'D', ('F','D'): 'D'}

def _greatest_common_dtype(alist):
    """
    Apply conversion rules to find the common conversion type
    Dtype 'd' is default for 'i' or unknown types
    (known types: 'f','d','F','D').
    """
    dtype = 'f'
    for array in alist:
        if array==None: continue
        tc = array.dtype.char
        if tc not in _symeig_pfx.keys(): tc = 'd'
        transition = (dtype, tc)
        if _type_conv.has_key(transition):
            dtype = _type_conv[transition]
    return dtype
    
# utilities for choosing the fortran routine with the optimal dtype

def get_froutines_func(name, prefix, dtype):
    """Return the optimal available froutines function for the
    selected dtype..
    The function returned has name prefix{common_type}+name.
    """
    # get the optimal function for the common conversion type    
    func_name = prefix[dtype] + name
    func = getattr(froutines,func_name)
    return func

